/*
 * LLRNet - network part of LLR
 *
 * (C) 2004-2005 Vincent Penne
 *
 * Released under GNU LIBRARY GENERAL PUBLIC LICENSE
 * (See file LICENSE that must be included with this software)
 *
 */

#define LLR_VERSION "3.5.0"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <signal.h>
#include <fcntl.h>

extern "C" {
#include "lua.h"
#include "lualib.h"
#include "lauxlib.h"

  int luaopen_luasqlmysql (lua_State *L);
  int luaopen_luasqlsqlite (lua_State *L);

#if defined(WIN32) && !defined(MSVC)
  void * __getreent() { return 0; }
  void * gethostname() { return 0; }
#endif
}
#include "bind-fltk.h"

#include "net.h"
#include "threads.h"

#ifndef LLRXX
extern "C" {
#endif
extern int VERBOSE;
#ifndef LLRXX
}
#endif


// Define an upper limit to the size of a string net_Recv can get.
// This will avoid hackers trying to fill the memory of the server 
// with insanly long messages.
#define RECV_MAX 256



static lua_State * L;
static lua_State * llr_L;

static Thrd_Mutex lua_mutex;
static void lock()
{
#ifdef THREADING
  lua_mutex.lock();
  //lua_settop(L, 0);
#endif
}
static void unlock()
{
#ifdef THREADING
  //lua_settop(L, 0);
  lua_mutex.unlock();
#endif
}

extern "C" void llrnet_lua_lock();
extern "C" void llrnet_lua_unlock();
void llrnet_lua_lock() { lock(); }
void llrnet_lua_unlock() { unlock(); }

#ifdef PTHREAD_MUTEX_RECURSIVE_NP
//PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP
pthread_mutexattr_t Thrd_Mutex_attrib = {PTHREAD_MUTEX_RECURSIVE_NP};
#endif

#ifndef WIN32
//# define DEBUG_SIGNALS
#endif

#ifdef DEBUG_SIGNALS
struct sigaction * signals_dump()
{
  int i;
  struct sigaction * act = new struct sigaction[64];
  for (i=0; i<64; i++)
    sigaction(i, 0, act + i);

  return act;
}

void signals_compare(struct sigaction * act1, struct sigaction * act2)
{
  int i;
  for (i=0; i<64; i++)
    if (memcmp(act1+i, act2+i, 4*sizeof(void *))) {
      printf("#%d [%x | %x] [%x | %x] [%x | %x]\n", i,
	     act1[i].sa_handler, act2[i].sa_handler, 
	     act1[i].sa_mask, act2[i].sa_mask, 
	     act1[i].sa_flags, act2[i].sa_flags);
    }
}
#endif

struct lua_thread_info {
  char function[128];
};

static void * lua_thread(void * arg)
{
  lua_thread_info * info = (lua_thread_info *) arg;

  lock();
  
  lua_pushlightuserdata(L, arg);
  lua_State * S = lua_newthread(L);
  lua_State * oldL = L; // keep this because L may change with the MFC GUI
  
  // get the function
  lua_pushlightuserdata(S, arg);
  lua_gettable(S, LUA_REGISTRYINDEX);

  // keep a reference on the thread (erasing function from the registry also)
  lua_settable(L, LUA_REGISTRYINDEX); 
  
  if (lua_resume(S, 0)) {
    fprintf(stderr, "lua_thread error : '%s'\n", lua_tostring(S, 1));
  }
  
  //lua_pop(L, 1);

  // remove reference on the thread so it can be garbage collected
  lua_pushlightuserdata(oldL, arg);
  lua_pushnil(oldL);
  lua_settable(oldL, LUA_REGISTRYINDEX);
  
  unlock();

  delete info;
  return 0;
}

static int lua_ScheduleThread(lua_State * S)
{
  unlock();
  lock();
  return 0;
}

static int lua_CreateThread(lua_State * S)
{
#ifdef THREADING
  Thrd_Thread id;
  int stack_size = (int) lua_tonumber(S, 2);
  lua_thread_info * info = new lua_thread_info;
  //strcpy(info->function, lua_tostring(S, 1)? lua_tostring(S, 1) : "?");
  lua_pushlightuserdata(S, info);
  lua_pushvalue(S, 1);
  lua_settable(S, LUA_REGISTRYINDEX);
  if (stack_size <= 0)
    stack_size = 64*1024;
  if(thrd_create_thread(id, lua_thread, info, stack_size) < 0) {
    fprintf(stderr, "thrd_create_thread failed\n");
    delete info;
    lua_settop(S, 0);
    lua_pushnumber(S, -1);
    return 1;
  }
#endif
  return 0;
}


class lua_Server_t : public net_Server_t {
 public:
  lua_Server_t(int maxConnections = 16) : net_Server_t(maxConnections) { }

  int OnConnect(int socket) {

    lock();
    
    lua_pushlightuserdata(L, &socket);
    lua_State * S = lua_newthread(L);
    lua_State * oldL = L; // keep this because L may change with the MFC GUI
    lua_settable(L, LUA_REGISTRYINDEX); // keep a reference on the thread
    
    //lua_xmove(L, S, -1); // move thread ref from main state to new thread

    lua_getglobal(S, "OnConnect");
    lua_pushnumber(S, socket);
    
    if (lua_resume(S, 1)) {
      fprintf(stderr, "OnConnect lua error : '%s'\n", lua_tostring(S, 1));
    }

    //lua_pop(L, 1);

    // remove reference on the thread so it can be garbage collected
    lua_pushlightuserdata(oldL, &socket);
    lua_pushnil(oldL);
    lua_settable(oldL, LUA_REGISTRYINDEX);

    unlock();

    net_Close(socket);

    return 0;
  }
};

static lua_Server_t * server;

void net_signal(int signal);
void net_InstallSignals()
{
  signal(SIGINT, net_signal);
  signal(SIGTERM, net_signal);
  signal(SIGABRT, net_signal);
#ifndef WIN32
//   signal(SIGPOLL, net_signal);
//   signal(SIGPROF, net_signal);
//   signal(SIGVTALRM, net_signal);
//   signal(SIGIO, net_signal);
//   signal(SIGHUP, net_signal);
//   signal(SIGALRM, net_signal);
//   signal(SIGUSR1, net_signal);
//   signal(SIGUSR2, net_signal);
  signal(SIGPIPE, net_signal);
//   signal(SIGURG, net_signal);
//   signal(SIGIO, net_signal);
  signal(SIGKILL, net_signal);
  signal(SIGQUIT, net_signal);
//   signal(SIGCHLD, net_signal);
#endif
}

struct server_info {
  char mask[128];
  char allow[128];
  int port;
};
#ifdef DEBUG_SIGNALS
static struct sigaction * main_signals;
#endif
static void * server_thread(void * arg)
{
  server_info * info = (server_info *) arg;

#ifdef DEBUG_SIGNALS
  struct sigaction * signals = signals_dump();
  signals_compare(signals, main_signals);
#endif

  //net_InstallSignals();
  server->Listen(info->port, info->mask, info->allow);

  if (server) {
    // delete is called by Listen now
    //delete server;
    server = 0;
  }
  fprintf(stderr, "Listening server on port %d closed\n", info->port);
  delete info;
  return 0;
}

static int lua_Server(lua_State * L)
{
  int maxconnections = (int) lua_tonumber(L, 1);
  int port = (int) lua_tonumber(L, 2);
  int res;
  char mask[128];
  char allow[128];
  const char * s;

  s = lua_tostring(L, 4);
  strncpy(mask, s? s : "0.0.0.0", sizeof(mask));
  mask[sizeof(mask)-1]=0;
  s = lua_tostring(L, 5);
  strncpy(allow, s? s : "0.0.0.0", sizeof(allow));
  allow[sizeof(allow)-1]=0;

  server = new lua_Server_t(maxconnections? maxconnections : 16);

  if (server) {
    if (lua_tonumber(L, 3)) {
#ifdef THREADING
      Thrd_Thread id;
      server_info * info = new server_info;
      strcpy(info->mask, mask);
      strcpy(info->allow, allow);
      info->port = port;
#ifdef DEBUG_SIGNALS
      main_signals = signals_dump();
#endif
      if(thrd_create_thread(id, server_thread, info) < 0) {
	fprintf(stderr, "thrd_create_thread failed\n");
	delete server;
	delete info;
	server = 0;
	return 1;
      }
      return 0;
#else
      fprintf(stderr, "thrd_create_thread failed\n(recompile llrnet with THREADING set)\n");
      return 1;
#endif
    }

    unlock();
    res = server->Listen(port, mask, allow);
    lock();
    if (server) {
      // delete is called by Listen now
      //delete server;
      server = 0;
    }

    if (res) {
      lua_pushnumber(L, res);
      return 1;
    }
  }

  return 0;
}


int lua_dummy(lua_State * S)
{
  return 0;
}

void lua_OutputStr(char * buf)
{
//   if (!llr_L) return;
//   lock();
//   lua_getglobal(llr_L, "OnPrint");
//   lua_pushstring(llr_L, buf);
//   lua_call(llr_L, 1, 0);

  if (!L) return;
  lock();
  lua_pushlightuserdata(L, &buf);
  lua_State * S = lua_newthread(L);
  lua_settable(L, LUA_REGISTRYINDEX);
  lua_getglobal(S, "OnPrint");
  lua_pushstring(S, buf);
  if (lua_resume(S, 1)) {
    fprintf(stderr, "OnPrint error : '%s'\n", lua_tostring(S, 1));
  }
  lua_pushlightuserdata(L, &buf);
  lua_pushnil(L);
  lua_settable(L, LUA_REGISTRYINDEX);
  unlock();
}

#ifndef MFCGUI
#ifndef NO_LLR
extern int MENUING;
extern int VERBOSE;

extern "C" void OutputStr (char *buf);
void OutputStr (char *buf)
{
  lua_OutputStr(buf);
  if (VERBOSE || MENUING) printf ("%s", buf);
}
#endif
#endif


static int lua_Connect(lua_State * L)
{
  int s;
  char * server = strdup(lua_tostring(L, 1));
  int port = (int) lua_tonumber(L, 2);
  int nonblock = (int) lua_tonumber(L, 3);

  unlock();
  s = net_Connect(server, port, nonblock);
  lock();

  free(server);

  if (s>=0) {
    lua_settop(L, 0);
    lua_pushnumber(L, s);
    return 1;
  }

  return 0;
}

static int lua_Close(lua_State * L)
{
  int s;

  s = (int) lua_tonumber(L, 1);

  net_Close(s);
  
  return 0;
}

typedef unsigned char u8_t;
struct header_t {
  char id[4];
  u8_t l1, l2, l3, l4;
};
static char * id = "LLR";

static int lua_Recv(lua_State * L)
{
  int s, l, cl;
  char * buffer;
  header_t hdr;
  char * p;
  int res;

  s = (int) lua_tonumber(L, 1);

  unlock();

  /* receive header */
  l = 0;
  p = (char *) &hdr;
  while (l<sizeof(hdr)) {
    res = net_Recv(s, p, sizeof(hdr) - l);
    if (res <= 0)
      goto error;
    
    p += res;
    l += res;
  }

  /* check header id */
  if (memcmp(id, hdr.id, sizeof(id))) {
    fprintf(stderr, "net_Recv : bad header '%c%c%c%c'\n", 
	    hdr.id[0],hdr.id[1],hdr.id[2],hdr.id[3]);
    goto qerror;
  }

  /* decode string size */
  l = int(hdr.l1) + 
    (int(hdr.l2)<<8) + 
    (int(hdr.l3)<<16) + 
    (int(hdr.l4)<<24);

#ifdef RECV_MAX
  if (l > RECV_MAX) {
    fprintf(stderr, "net_Recv : received a too long message (len = %d, socket = %d), probably\n someone is trying to hack onto your computer. No worries, the connection\n will be closed shortly with no harm done.\n", l, s);
    goto qerror;
  }
#endif

  /* read in the string */
  buffer = (char *) malloc(l);
  cl = 0;
  p = buffer;
  while (cl < l) {
    res = net_Recv(s, p, l-cl);
    if (res <= 0) {
      free(buffer);
      goto error;
    }

    cl += res;
    p += res;
  }

  lock();

  lua_settop(L, 0);
  lua_pushlstring(L, buffer, l);
  
  free(buffer);

  return 1;

 error:
  fprintf(stderr, "recv error res=%d, errno=%d\n", res, errno);
 qerror:

  lock();
  //exit(1);
  return 0;
}

// As sending should not be a blocking operation, it is not necessary to release
// the mutex. If on some platform it happened that send may block, then one should
// uncomment the line below.
// UPDATE : actually it would be a bad idea since the lua code rely on the fact
// that a send operation is *atomic*.
//#define LOCK_SEND

static int lua_SendTable(lua_State * L)
{
  int s, l, n, i;
  const char * string;
  char * buffer = 0, * ptr;
  header_t * hdr;

  s = (int) lua_tonumber(L, 1);

  l = n = 0;
  for ( ; ; ) {
    lua_rawgeti(L, 2, n+1);
    if (!lua_isstring(L, -1)) {
      lua_pop(L, 1);
      break;
    }
    l += lua_strlen(L, -1);
    lua_pop(L, 1);
    n++;
  }

  ptr = buffer = (char *) malloc(n*sizeof(header_t) + l);
  if (!buffer)
    return 0;

  for (i=0; i<n; i++) {
    int l;
    lua_rawgeti(L, 2, i+1);
    string = lua_tostring(L, -1);
    l = lua_strlen(L, -1);
    lua_pop(L, 1);

    hdr = (header_t *) ptr;
    memcpy(hdr->id, id, sizeof(id));
    hdr->l1 = l&0xff;
    hdr->l2 = (l>>8)&0xff;
    hdr->l3 = (l>>16)&0xff;
    hdr->l4 = (l>>24)&0xff;

    memcpy(ptr + sizeof(header_t), string, l);
    ptr += sizeof(header_t) + l;
  }
  
#ifdef LOCK_SEND  
  unlock();
#endif

  /* send buffer */
  if (n*sizeof(header_t) + l == net_Send(s, buffer, n*sizeof(header_t) + l)) {
    free(buffer);
#ifdef LOCK_SEND
    lock();
#endif
    lua_settop(L, 0);
    lua_pushnumber(L, l);
    return 1;
  }

  free(buffer);
#ifdef LOCK_SEND
  lock();
#endif
  return 0;
}

static int lua_Send(lua_State * L)
{
  int s, l;
  const char * string;
  char * buffer = 0;
  header_t * hdr;

  s = (int) lua_tonumber(L, 1);
  string = lua_tostring(L, 2);
  l = lua_strlen(L, 2);

  if (l<0)
    return 0;

#ifdef LOCK_SEND  
  unlock();
#endif

  buffer = (char *) malloc(sizeof(header_t) + l);

  hdr = (header_t *) buffer;
  memcpy(hdr->id, id, sizeof(id));
  hdr->l1 = l&0xff;
  hdr->l2 = (l>>8)&0xff;
  hdr->l3 = (l>>16)&0xff;
  hdr->l4 = (l>>24)&0xff;

  memcpy(buffer + sizeof(header_t), string, l);

  /* send buffer */
  if (sizeof(header_t) + l == net_Send(s, buffer, sizeof(header_t) + l)) {
    if (buffer) free(buffer);
#ifdef LOCK_SEND
    lock();
#endif
    lua_settop(L, 0);
    lua_pushnumber(L, l);
    return 1;
  }

 error:
  if (buffer) free(buffer);
#ifdef LOCK_SEND
  lock();
#endif
  return 0;
}

static int lua_NonBlock(lua_State * L)
{
  int socket = (int) lua_tonumber(L, 1);
  int mode = (int) lua_tonumber(L, 2);
  unlock();
  int res = net_NonBlock(socket, mode);
  lock();
  return res? 1:0;
}

static int lua_sleep(lua_State * L)
{
  int t = (int) lua_tonumber(L, 1);

  unlock();
  thrd_sleep(t);
  lock();

  return 0;
}

static void lua_net_init(lua_State * L)
{
  lua_register(L, "net_Close", lua_Close);
  lua_register(L, "net_Connect", lua_Connect);
  lua_register(L, "net_NonBlock", lua_NonBlock);
  lua_register(L, "net_Send", lua_Send);
  lua_register(L, "net_SendTable", lua_SendTable);
  lua_register(L, "net_Recv", lua_Recv);
  lua_register(L, "net_Server", lua_Server);

  lua_register(L, "sleep", lua_sleep);
}

void debug_signal_handler(int signal)
{
  printf("DEBUG SIGNAL #%d\n", signal);
  exit(0);
}

void net_signal(int signal)
{
  static Thrd_Mutex mutex;

#ifndef WIN32
  if (signal == SIGCHLD) {
    fprintf(stderr, "SIGCHLD catched\n");
    return;
  }
  if (signal == SIGPIPE) {
    fprintf(stderr, "broken pipe\n");
    return;
  }
  if (signal == SIGIO) {
    fprintf(stderr, "SIGIO signaled\n");
    return;
  }
  if (signal == SIGURG) {
    fprintf(stderr, "SIGURG signaled\n");
    return;
  }
#endif

  fprintf(stderr, "net_signal called with code %d\n", signal);
  mutex.lock();

  if (server) {
    //lock();
    lua_Server_t * s = server;
    if (server) {
      server = 0;

      s->Kill();

      // we do not delete here anymore because server code need to clean up
      // properly.
      // delete is called into net_Server::Listen
      //delete s;
    }
  }

#ifndef WIN32
  net_Shutdown();
  
  exit(0);
#endif
}

#ifndef NO_LLR
/* defined in llr.c */
#ifndef LLRXX
extern "C" {
#endif
int primeTest(const char * type, const char * input, char * residue);
int llrInit(lua_State * L);
#ifndef LLRXX
}
#endif

int lua_primeTest(lua_State * L)
{
  char residue[17];
  int retval;
  const char * type;
  const char * input;

  llr_L = L;
  
  type = lua_tostring(L, 1);
  input = lua_tostring(L, 2);
  residue[0] = 0;

  unlock();
  retval = primeTest(type, input, residue);
  lock();

  lua_settop(L, 0);
  lua_pushnumber(L, retval);
  lua_pushstring(L, residue);

  return 2;
}

int lua_llrVerbose(lua_State * L)
{
  VERBOSE = (int) lua_tonumber(L, 1);

  return 0;
}
#endif

int lua_detach(lua_State * L)
{
#ifndef WIN32
/* To enter daemon mode, close all the filedescs and detach from the tty */
  int	fd;
  fd = fork ();
  if (fd == -1) {
    perror ("Could not fork to the background");
    exit (1);
  }
  if (fd != 0) {
    FILE * fp = fopen("llrnet.pid", "w");
    if (fp) {
      fprintf(fp, "%d", fd);
      fclose(fp);
    }

    printf("pid = %d\n", fd);
  
    exit (0);
  }

#if 1
  for (fd = 0; fd < 3; fd++) {
    close (fd);
  }
  open ("/dev/null", O_APPEND);
  dup2 (0,1);
  dup2 (0,2);
#endif

  setsid ();
#endif
  return 0;
}

/* run a command as separate process */
int lua_bsystem(lua_State * L)
{
#ifndef WIN32
/* To enter daemon mode, close all the filedescs and detach from the tty */
  int	fd;
  fd = fork ();
  if (fd == -1) {
    perror ("Could not fork");
    return 0;
  }
  if (fd != 0) {
    printf("pid = %d\n", fd);
    lua_settop(L, 0);
    lua_pushnumber(L, fd);
    return 1;
  }

  //setsid ();
  static char * array[10];
  int i;
  for (i=1; i<10 && i<=lua_gettop(L); i++)
    array[i-1] = (char *) lua_tostring(L, i);
  array[i-1] = 0;

  execv(array[0], array);

  exit(0);
#endif
  return 0;
}

#ifndef WIN32
# include <sys/wait.h>
#endif
int lua_waitpid(lua_State * L)
{
#ifndef WIN32
  int status;
  int pid = (int) lua_tonumber(L, 1);
  int res;
  unlock();
  res = waitpid(pid, &status, __WALL);
  lock();
  lua_settop(L, 0);
  lua_pushnumber(L, res);
#endif
  return 1;
}

#if defined(WIN32)
// windoz specific stuffs
#include <shellapi.h>
#include <wincon.h>
#include <winuser.h>

#ifndef ATTACH_PARENT_PROCESS
# define ATTACH_PARENT_PROCESS ((DWORD)-1)
#endif

typedef BOOL (WINAPI * AttachConsoleType)(DWORD);
BOOL (WINAPI * AttachConsolePTR)(DWORD);

static HINSTANCE hInstance;

static LRESULT CALLBACK LLRwndProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
//   switch (message) {
//   case WM_DESTROY: {
//     NOTIFYICONDATA nid;
//     nid.cbSize = sizeof(NOTIFYICONDATA);
//     nid.hWnd = hwnd;
//     nid.uID = 0;
//     nid.uFlags = 0;
//     Shell_NotifyIcon(NIM_DELETE, &nid);
//     PostQuitMessage(0);
//   }
//   }

  lock();

  lua_State * oldL = L;

  lua_pushlightuserdata(L, &hwnd);
  lua_State * S = lua_newthread(L);
  lua_settable(L, LUA_REGISTRYINDEX); // keep a reference on the thread
  lua_getglobal(S, "trayCB"); 
  lua_pushlightuserdata(S, hwnd);
  lua_pushnumber(S, message);
  lua_pushnumber(S, wParam);
  lua_pushnumber(S, lParam);
  if (lua_resume(S, 4)) {
    fprintf(stderr, "trayCB error : '%s'\n", lua_tostring(S, 1));
  }

  lua_pushlightuserdata(oldL, &hwnd);
  lua_pushnil(oldL);
  lua_settable(oldL, LUA_REGISTRYINDEX); // remove reference on the thread
  
  unlock();
  return 1;
}

static int lua_TrayIconRemove(lua_State * L)
{
  Shell_NotifyIcon(NIM_DELETE, (PNOTIFYICONDATA) lua_touserdata(L, 1));
  return 0;
}

static int lua_TrayIconTip(lua_State * L)
{
  PNOTIFYICONDATA icon;

  icon = (PNOTIFYICONDATA) lua_touserdata(L, 1);
  
  strncpy(icon->szTip, lua_tostring(L, 2), 63);
  icon->uFlags = NIF_TIP;
  Shell_NotifyIcon(NIM_MODIFY, icon);
  return 0;
}

static int lua_TrayIconAdd(lua_State * L)
{
  PNOTIFYICONDATA icon;

  static HWND hwnd;
  static WNDCLASSEX wc;
  const char * classname = lua_tostring(L, 1);

  if (!hwnd) {
    wc.cbSize = sizeof(WNDCLASSEX);
    wc.style = 0;
    wc.lpfnWndProc = LLRwndProc;
    wc.cbClsExtra = wc.cbWndExtra = 0;
    wc.hInstance = hInstance;
    wc.hIcon = NULL;
    wc.hCursor = NULL;
    wc.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
    wc.lpszMenuName = NULL;
    wc.lpszClassName = classname;
    wc.hIconSm = NULL;
    RegisterClassEx(&wc);
    
    // Create window. Note that WS_VISIBLE is not used, and window is never shown
    hwnd = CreateWindowEx(0, classname, classname, WS_POPUP, CW_USEDEFAULT, 0,
			  CW_USEDEFAULT, 0, NULL, NULL, hInstance, NULL);
    typedef BOOL (*WTSRegisterSessionNotificationTYPE)(HWND hWnd, DWORD dwFlags);
    BOOL (*WTSRegisterSessionNotificationPTR)(HWND hWnd, DWORD dwFlags);

    HANDLE libhnd;
    (libhnd = LoadLibrary("WTSAPI32")) && 
      (WTSRegisterSessionNotificationPTR = (WTSRegisterSessionNotificationTYPE)GetProcAddress((HINSTANCE__*)libhnd, "WTSRegisterSessionNotification")) && 
      WTSRegisterSessionNotificationPTR(hwnd, 1/*NOTIFY_FOR_ALL_SESSION*/);
  }

  icon = new NOTIFYICONDATA;

  memset(icon, 0, sizeof(*icon));
  icon->cbSize = sizeof(*icon);
  icon->hWnd = hwnd;
  //icon->hWnd = (HWND) lua_touserdata(L, 1);
  //printf("hwnd = %x\n", icon->hWnd);
  icon->uID = (int) lua_tonumber(L, 2);
  icon->hIcon = (HICON)lua_touserdata(L, 3);
  static BYTE buf[1024];
  memset(buf, 0xdeadbeef, sizeof(buf));
  icon->hIcon = CreateIcon(hInstance, 32, 32, 1, 1, buf, buf);
  icon->uCallbackMessage = WM_USER + 1;
  strncpy(icon->szTip, lua_tostring(L, 4), 63);
  icon->uFlags = NIF_ICON | NIF_MESSAGE | NIF_TIP;

  if (Shell_NotifyIcon(NIM_ADD, icon)) {
    lua_settop(L, 0);
    lua_pushlightuserdata(L, icon);
    return 1;
  } else {
    delete icon;
    return 0;
  }
}

static int lua_GetCursorPos(lua_State * L)
{
  POINT pt;
  GetCursorPos(&pt);
  lua_settop(L, 0);
  lua_pushnumber(L, pt.x);
  lua_pushnumber(L, pt.y);
  return 2;
}

static int lua_SetForegroundWindow(lua_State * L)
{
  SetForegroundWindow((HWND)lua_touserdata(L, 1));
  return 0;
}

static int lua_ShellExecute(lua_State * L)
{
  ShellExecute((HWND)lua_touserdata(L, 1),
	       lua_tostring(L, 2), lua_tostring(L, 3), 
	       lua_tostring(L, 4), lua_tostring(L, 5), 
	       (int) lua_tonumber(L, 6));
  return 0;
}

static int lua_CreatePopupMenu(lua_State * L)
{
  lua_settop(L, 0);
  lua_pushlightuserdata(L, CreatePopupMenu());
  return 1;
}

static int lua_InsertMenuItem(lua_State * L)
{
  HMENU menu = (HMENU) lua_touserdata(L, 1);
  MENUITEMINFO info;

  memset(&info, 0, sizeof(info));
  info.cbSize = sizeof(info);
  info.fMask = MIIM_FTYPE/*|MIIM_TYPE*/|MIIM_STRING|MIIM_ID;
  info.fType = MFT_STRING;
  info.wID = (int) lua_tonumber(L, 5);
  info.dwTypeData = (char *) lua_tostring(L, 4);
  info.cch = strlen(info.dwTypeData);
  InsertMenuItem(menu, (int) lua_tonumber(L, 2), (int) lua_tonumber(L, 3), 
		 &info);
  return 0;
}

static int lua_PopupMenu(lua_State * L)
{
  POINT pt;
  HMENU hmenu;
  HMENU hpopup;
  HWND hwnd;
  
  hwnd = (HWND) lua_touserdata(L, 1);
  hmenu = (HMENU) lua_touserdata(L, 2);

  GetCursorPos(&pt);
  hpopup = hmenu;//GetSubMenu(hmenu, 0);

  /* SetForegroundWindow and the ensuing null PostMessage is a
     workaround for a Windows 95 bug (see MSKB article Q135788,
     http://www.microsoft.com/kb/articles/q135/7/88.htm, I think).
     In typical Microsoft style this bug is listed as "by design".
     SetForegroundWindow also causes our MessageBox to pop up in front
     of any other application's windows. */
  SetForegroundWindow(hwnd);
  /* We specifiy TPM_RETURNCMD, so TrackPopupMenu returns the menu
     selection instead of returning immediately and our getting a
     WM_COMMAND with the selection. You don't have to do it this way.
  */
  unlock();
  int res = TrackPopupMenu(hpopup,            // Popup menu to track
			   TPM_RETURNCMD |    // Return menu code
			   TPM_RIGHTBUTTON,   // Track right mouse button?
			   pt.x, pt.y,        // screen coordinates
			   0,                 // reserved
			   hwnd,              // owner
			   NULL);             // LPRECT user can click in
                                              // without dismissing menu
  lock();
  PostMessage(hwnd, 0, 0, 0); // see above
  DestroyMenu(hmenu); // Delete loaded menu and reclaim its resources

  lua_settop(L, 0);
  lua_pushnumber(L, res);
  return 1;
}

static int lua_WinLoop(lua_State * L)
{
  MSG msg;
  unlock();

  while (GetMessage(&msg, NULL, 0, 0)) {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  }
  
  lock();
  return 0;
}

static int HasConsole = 0;

static BOOL WINAPI ConsoleHandler(DWORD dwCtrlType)
{
  if (/*dwCtrlType == CTRL_CLOSE_EVENT || */dwCtrlType == CTRL_LOGOFF_EVENT) {
    HasConsole = 0;
    FreeConsole();
    return 1;
  }
  raise(SIGINT);
  return 1;
}

static int lua_AllocConsole(lua_State * L)
{
  AllocConsole();
  HasConsole = 1;
  SetConsoleTitle(lua_tostring(L, 1));
  //SetConsoleCtrlHandler(NULL, 1);
  SetConsoleCtrlHandler(ConsoleHandler, 1);
  FILE * newstdout = freopen("CON", "w", stdout);
  //_dup2(_fileno(stdout), _fileno(stderr));
  FILE * newstderr = freopen("CON", "w", stderr);
  FILE * newstdin = freopen("CON", "w", stdin);
  return 0;
}

static int lua_FreeConsole(lua_State * L)
{
  HasConsole = 0;
  FreeConsole();
  
  if (lua_tonumber(L, 1)) {
    FILE * newstdout = freopen("stdout.txt", "a", stdout);
    _dup2(_fileno(stdout), _fileno(stderr));
  }
  return 0;
}

static int lua_HasConsole(lua_State * L)
{
  lua_settop(L, 0);
  lua_pushnumber(L, HasConsole);
  return 1;
}

static int lua_OpenSCManager(lua_State * L)
{
  SC_HANDLE h;

  h = OpenSCManager(lua_tostring(L, 1), lua_tostring(L, 2), SC_MANAGER_ALL_ACCESS);
  if (h) {
    lua_settop(L, 0);
    lua_pushlightuserdata(L, h);
    return 1;
  } else {
    fprintf(stderr, "OpenSCManager failed (%d)\n", GetLastError()); 
    return 0;
  }
}

static int lua_CloseServiceHandle(lua_State * L)
{
  CloseServiceHandle((SC_HANDLE) lua_touserdata(L, 1));
  return 0;
}

static int lua_DeleteService(lua_State * L)
{
  DeleteService((SC_HANDLE) lua_touserdata(L, 1));
  return 0;
}

static int lua_CreateService(lua_State * L)
{
  SC_HANDLE manager = (SC_HANDLE) lua_touserdata(L, 1);
  SC_HANDLE h;
  TCHAR path[MAX_PATH]; 
  
  if( !GetModuleFileName( NULL, path, MAX_PATH ) ) {
    fprintf(stderr, "GetModuleFileName failed (%d)\n", GetLastError()); 
    return 0;
  }

  h = CreateService(manager, lua_tostring(L, 2), lua_tostring(L, 3),
		    SERVICE_ALL_ACCESS,	// desired access
		    SERVICE_INTERACTIVE_PROCESS |
		    SERVICE_WIN32_OWN_PROCESS,  // service type
		    SERVICE_AUTO_START,	// start type
		    SERVICE_ERROR_NORMAL,
		    path, NULL, NULL, NULL, NULL, NULL);

  if (h) {
    lua_settop(L, 0);
    lua_pushlightuserdata(L, h);
    return 1;
  } else {
    fprintf(stderr, "CreateService failed (%d)\n", GetLastError()); 
    return 0;
  }
}

static int lua_OpenService(lua_State * L)
{
  SC_HANDLE manager = (SC_HANDLE) lua_touserdata(L, 1);
  SC_HANDLE h;
  
  h = OpenService(manager, lua_tostring(L, 2), SERVICE_ALL_ACCESS);

  if (h) {
    lua_settop(L, 0);
    lua_pushlightuserdata(L, h);
    return 1;
  } else {
    //fprintf(stderr, "OpenService failed (%d)\n", GetLastError()); 
    return 0;
  }
}

// Changed to use RegisterServiceCtrlHandler
void /*DWORD*/ WINAPI LLRserviceProc(
  DWORD dwControl/*,
  DWORD dwEventType,
  LPVOID lpEventData,
  LPVOID lpContext*/)
{
  if (!L) {
    return/* NO_ERROR*/;
  }
  lock();

  lua_State * oldL = L;

  lua_pushlightuserdata(L, &dwControl);
  lua_State * S = lua_newthread(L);
  lua_settable(L, LUA_REGISTRYINDEX); // keep a reference on the thread
  lua_getglobal(S, "serviceCB"); 
  lua_pushnumber(S, dwControl);
/*  lua_pushnumber(S, dwEventType);
  lua_pushlightuserdata(S, lpEventData);
  if (lua_resume(S, 3)) {*/
  if (lua_resume(S, 1)) {
    fprintf(stderr, "serviceCb error : '%s'\n", lua_tostring(S, 1));
  }

  lua_pushlightuserdata(oldL, &dwControl);
  lua_pushnil(oldL);
  lua_settable(oldL, LUA_REGISTRYINDEX); // remove reference on the thread
  
  unlock();

  return/* NO_ERROR*/;
}

//#define SERVICE_DEBUG

static SERVICE_STATUS_HANDLE sshnd;

extern "C" VOID WINAPI ServiceMain(DWORD dwArgc, LPTSTR* lpszArgv);
VOID WINAPI ServiceMain(DWORD dwArgc,LPTSTR* lpszArgv)
{
#ifdef SERVICE_DEBUG
  AllocConsole();
  FILE * newstdout = freopen("CON", "w", stdout);
  FILE * newstderr = freopen("CON", "w", stderr);
  FILE * newstdin = freopen("CON", "w", stdin);
#endif

  if (dwArgc > 0) {
    // Changed to use RegisterServiceCtrlHandler
    sshnd = RegisterServiceCtrlHandler(lpszArgv[0], LLRserviceProc);
    //sshnd = RegisterServiceCtrlHandlerEx(lpszArgv[0], LLRserviceProc, NULL);
    if (sshnd) {
      SERVICE_STATUS status;
      memset(&status, 0, sizeof(status));
      status.dwServiceType = SERVICE_WIN32_OWN_PROCESS | SERVICE_INTERACTIVE_PROCESS;
      status.dwCurrentState = SERVICE_RUNNING;
      status.dwControlsAccepted = SERVICE_ACCEPT_SHUTDOWN | SERVICE_ACCEPT_STOP;
      SetServiceStatus(sshnd, &status);
    }
  }
}

static int lua_SetServiceStatus(lua_State * L)
{
  if (!sshnd)
    return 0;

  SERVICE_STATUS status;
  memset(&status, 0, sizeof(status));
  status.dwServiceType = SERVICE_WIN32_OWN_PROCESS | SERVICE_INTERACTIVE_PROCESS;
  status.dwCurrentState = (int) lua_tonumber(L, 1);
  status.dwControlsAccepted = SERVICE_ACCEPT_SHUTDOWN | SERVICE_ACCEPT_STOP;
  SetServiceStatus(sshnd, &status);

  return 1;
}

static int lua_RegisterServiceCtrlHandler(lua_State * L)
{
  static SERVICE_TABLE_ENTRY table[] = {
    { "LLRnet", ServiceMain, },
    { 0, 0 },
  };

  table[0].lpServiceName = strdup(lua_tostring(L, 1));
  unlock();
  int res = StartServiceCtrlDispatcher(table);
  fprintf(stderr, "StartServiceCtrlDispatcher returns %d\n", res);
  lock();

  return 0;

  SERVICE_STATUS_HANDLE h;
  //h = RegisterServiceCtrlHandlerEx(lua_tostring(L, 1), LLRserviceProc, NULL);
  h = sshnd;
  if (h) {
    lua_settop(L, 0);
    lua_pushlightuserdata(L, (void *) h);
    return 1;
  } else {
    fprintf(stderr, "RegisterServiceCtrlHandler failed (%d)\n", GetLastError()); 
    return 0;
  }
}

static int lua_InitWin32(lua_State * L)
{
  hInstance = GetModuleHandle(NULL);
  lua_register(L, "TrayIconTip", lua_TrayIconTip);
  lua_register(L, "TrayIconAdd", lua_TrayIconAdd);
  lua_register(L, "TrayIconRemove", lua_TrayIconRemove);
  lua_register(L, "GetCursorPos", lua_GetCursorPos);
  lua_register(L, "SetForegroundWindow", lua_SetForegroundWindow);
  lua_register(L, "CreatePopupMenu", lua_CreatePopupMenu);
  lua_register(L, "InsertMenuItem", lua_InsertMenuItem);
  lua_register(L, "PopupMenu", lua_PopupMenu);
  lua_register(L, "WinLoop", lua_WinLoop);
  lua_register(L, "ShellExecute", lua_ShellExecute);
  lua_register(L, "AllocConsole", lua_AllocConsole);
  lua_register(L, "FreeConsole", lua_FreeConsole);
  lua_register(L, "HasConsole", lua_HasConsole);
  lua_register(L, "OpenSCManager", lua_OpenSCManager);
  lua_register(L, "CreateService", lua_CreateService);
  lua_register(L, "DeleteService", lua_DeleteService);
  lua_register(L, "CloseServiceHandle", lua_CloseServiceHandle);
  lua_register(L, "OpenService", lua_OpenService);
  lua_register(L, "SetServiceStatus", lua_SetServiceStatus);
  lua_register(L, "RegisterServiceCtrlHandler", lua_RegisterServiceCtrlHandler);

  lua_pushnumber(L, SERVICE_CONTROL_SHUTDOWN);
  lua_setglobal(L, "SERVICE_CONTROL_SHUTDOWN");
  lua_pushnumber(L, SERVICE_CONTROL_STOP);
  lua_setglobal(L, "SERVICE_CONTROL_STOP");
  lua_pushnumber(L, SERVICE_RUNNING);
  lua_setglobal(L, "SERVICE_RUNNING");
  lua_pushnumber(L, SERVICE_STOP_PENDING);
  lua_setglobal(L, "SERVICE_STOP_PENDING");
  lua_pushnumber(L, SERVICE_STOPPED);
  lua_setglobal(L, "SERVICE_STOPPED");

  lua_pushnumber(L, 0x2B1/*WM_WTSSESSION_CHANGE*/);
  lua_setglobal(L, "WM_WTSSESSION_CHANGE");
  lua_pushnumber(L, 0x5/*WTS_SESSION_LOGON*/);
  lua_setglobal(L, "WTS_SESSION_LOGON");
  lua_pushnumber(L, 0x6/*WTS_SESSION_LOGOFF*/);
  lua_setglobal(L, "WTS_SESSION_LOGOFF");

  return 0;
}
#endif

static int lua_system(lua_State * L)
{
  const char * cmd = lua_tostring(L, 1);
  unlock();
  system(cmd);
  lock();
  return 0;
}

static int lua_raise(lua_State * L)
{
  int num = (int) lua_tonumber(L, 1);
  unlock();
  raise(num);
  lock();
  return 0;
}

int lua_SemaCreate(lua_State * L)
{
  int res = thrd_SemaCreate((int) lua_tonumber(L, 1), (int) lua_tonumber(L, 2), 0);
  if (res >= 0) {
    lua_settop(L, 0);
    lua_pushlightuserdata(L, (void *) res);
    return 1;
  } else
    return 0;
}

int lua_SemaDelete(lua_State * L)
{
  int s = (int) lua_touserdata(L, 1);
  unlock();
  thrd_SemaDelete(s);
  lock();
  return 0;
}

int lua_SemaWait(lua_State * L)
{
  int s = (int) lua_touserdata(L, 1);
  unlock();
  thrd_SemaWait(s);
  lock();
  return 0;
}

int lua_SemaSignal(lua_State * L)
{
  int s = (int) lua_touserdata(L, 1);
  unlock();
  thrd_SemaSignal(s);
  lock();
  return 0;
}

#ifdef USE_FOX
int g_argc;
char * * g_argv;
int init_lua_fox_State(lua_State * _S);
#endif

int main(int argc, char * * argv)
{
  int i;

#ifdef WIN32
  //AllocConsole();

/* Change the working directory to the same directory that */
/* the executable is located.  This is especially important */
/* for running prime95 as a Windows 95 service */
  {
    char	buf[256];
    GetModuleFileName (NULL, buf, sizeof (buf));
    strrchr (buf, '\\')[1] = 0;
    _chdir (buf);
  }
  
  HANDLE libhnd;
  if ((libhnd = LoadLibrary("KERNEL32")) && (AttachConsolePTR = (AttachConsoleType)GetProcAddress((HINSTANCE__*)libhnd, "AttachConsole")) && AttachConsolePTR(ATTACH_PARENT_PROCESS)) 
    {
      HasConsole = 1;
      FILE * newstdout = freopen("CON", "w", stdout);
      FILE * newstderr = freopen("CON", "w", stderr);
      FILE * newstdin = freopen("CON", "r", stdin);
  }/* else {
    FILE * newstdout = freopen("stdout.txt", "w", stdout);
    _dup2(_fileno(stdout), _fileno(stderr));
    //FILE * newstderr = freopen("stderr.txt", "w", stderr);
    }*/
//   _dup2(0, _fileno(stdout));
//   _dup2(1, _fileno(stdin));
//   _dup2(2, _fileno(stderr));

  SetConsoleCtrlHandler(ConsoleHandler, 1);

#endif

  lua_State * S = lua_open();

  lua_baselibopen(S);
  lua_tablibopen(S);
  lua_strlibopen(S);
  lua_mathlibopen(S);
  lua_iolibopen(S);

#ifdef USE_FOX
  g_argc = argc;
  g_argv = argv;
  
//   if (init_lua_fox_State(S)) {
//     printf("Error initialising fox lua binding\n");
//     return -1;
//   }
  lua_register(S, "fox_init", init_lua_fox_State);

#endif

#ifdef USE_FLTK
  if (init_lua_fltk_State(S))
    return -1;
#endif

#ifdef USE_MYSQL
  luaopen_luasqlmysql(S);
#endif
#ifdef USE_SQLITE
  luaopen_luasqlsqlite(S);
#endif

  lua_settop(S, 0);
#ifdef WIN32
  lua_InitWin32(S);
  lua_pushnumber(S, 1);
  lua_setglobal(S, "WIN32");
  lua_settop(S, 0);
#endif
#ifdef MFC
  lua_pushnumber(S, 1);
  lua_setglobal(S, "MFC");
  lua_settop(S, 0);
#endif

  lua_newtable(S);
  for (i=0; i<argc; i++) {
    lua_pushstring(S, argv[i]);
    lua_rawseti(S, 1, i);
  }
  lua_setglobal(S, "arg");

  char s[256];
  char * p = s;
  strcpy(s, argv[0]);
  p += strlen(s);
  while (p>s && p[-1] != '/' && p[-1] != '\\')
    p--;

  *p = 0;
  lua_pushstring(S, s);
  lua_setglobal(S, "EXEDIR");

  if (net_Init())
    return -1;

//   {
//     int i;
//     for (i=0; i<64; i++)
//       signal(i, debug_signal_handler);
//   }
  net_InstallSignals();

  lua_net_init(S);

#ifndef NO_LLR
  //llrInit(S);
  lua_register(S, "llrInit", llrInit);
  lua_register(S, "primeTest", lua_primeTest);
  lua_register(S, "llrVerbose", lua_llrVerbose);
#endif

  lua_register(S, "detach", lua_detach);
  lua_register(S, "system", lua_system);
  lua_register(S, "bsystem", lua_bsystem);
  lua_register(S, "waitpid", lua_waitpid);
  lua_register(S, "raise", lua_raise);
  lua_pushnumber(S, SIGINT);
  lua_setglobal(S, "SIGINT");
  lua_pushnumber(S, SIGTERM);
  lua_setglobal(S, "SIGTERM");
  lua_pushnumber(S, SIGABRT);
  lua_setglobal(S, "SIGABRT");

  lua_register(S, "SemaCreate", lua_SemaCreate);
  lua_register(S, "SemaDelete", lua_SemaDelete);
  lua_register(S, "SemaWait", lua_SemaWait);
  lua_register(S, "SemaSignal", lua_SemaSignal);

  lua_register(S, "schedule_thread", lua_ScheduleThread);
  lua_register(S, "create_thread", lua_CreateThread);

  lua_pushstring(S, LLR_VERSION);
  lua_setglobal(S, "LLRversion");

  lock();

  lua_register(S, "OnPrint", lua_dummy);

  L = S;
  llr_L = S;

  strcpy(p, DEFAULT_LUA_FILE);

  for (i=1; i<argc; i++) {
    if (argv[i][0] == '-')
      break;
    strcpy(p, argv[i]);
  }

  lua_dofile(L, s);

  L = 0;
  unlock();
  //lua_close(S);

  net_Shutdown();
  
  return 0;
}

