/*
 * LLRNet - network part of LLR
 *
 * (C) 2004-2005 Vincent Penne
 *
 * Released under GNU LIBRARY GENERAL PUBLIC LICENSE
 * (See file LICENSE that must be included with this software)
 *
 */

#include <sys/types.h>
#ifdef WIN32
# include <windows.h>
# ifndef FD_SET
extern "C" {
#  include <winsock2.h>
}
# endif
# include <assert.h>
# define SHUT_RDWR SD_BOTH
# define socklen_t int
# define close closesocket
# define ioctl ioctlsocket
int errno;
#else
# include <sys/socket.h>
# include <netinet/in.h>
# include <netdb.h>
# include <unistd.h>
# include <sys/ioctl.h>
#endif
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <errno.h>

#include "net.h"

#ifdef WIN32
WORD wVersionRequested;
WSADATA wsaData;
#endif

static int init;

int net_Init()
{
  if (init)
    return 0;

#ifdef WIN32
  wVersionRequested = MAKEWORD( 2, 2 ); 

  int err = WSAStartup( wVersionRequested, &wsaData );

  if ( err ||
       LOBYTE( wsaData.wVersion ) != 2 || HIBYTE( wsaData.wVersion ) != 2 ) {
    /* Tell the user that we could not find a usable */
    /* WinSock DLL.                                  */    
      
    printf("no usuable version of winsock dll found\n");
      
    WSACleanup( );
    return -1; 
  }		
#endif

  init = 1;

  return 0;
}


int net_Shutdown()
{

  if (!init)
    return 0;

#ifdef WIN32
  WSACleanup( );
#endif

  init = 0;

  return 0;
}






net_Server_t::net_Server_t(int maxConnections)
{
  Init(maxConnections);
}

net_Server_t::~net_Server_t()
{
  Kill();
}

int net_Server_t::Init(int pmaxConnections)
{
  maxConnections = pmaxConnections;
  s = -1;

  return 0;
}
void net_Server_t::Kill()
{
  if (s > 0) {
    fprintf(stderr, "shutting down listening socket\n");
#ifndef WIN32
    shutdown(s, SHUT_RDWR);
#endif
    close(s);
    s = -1;
  }
}

void * net_Server_t::connection_thread(void * arg)
{
  threadPool_t * thd = (threadPool_t *) arg;

  while (!thd->quit) {
    thrd_SemaWait(thd->sema);
    
    if (thd->quit) {
      net_Close(thd->socket);
      thd->socket = -2;
      break;
    }

    assert(thd->socket >= 0);

    thd->server->OnConnect(thd->socket);

    printf("connection closed (socket %d)\n", thd->socket);

#ifdef WIN32xxx
    thd->server = 0;
    thd->socket = -1;
    return 0;
#endif
    thd->socket = -1;
  }

  fprintf(stderr, "thread #%d exited\n", thd - thd->server->threadPool);

  thd->quit = 0;

  return 0;
}

int gethost(const char * name, struct sockaddr_in * sin)
{
#ifdef WIN32
  u_int temp;
  if (!strcmp(name, "255.255.255.255")) {
    sin->sin_family = AF_INET;
    sin->sin_addr.s_addr = ~0;
    return 0;
  }
  if((u_int)-1 != (temp = inet_addr(name))){
    sin->sin_family = AF_INET;
    sin->sin_addr.s_addr = temp;
    return 0;
  }
#endif
  
  struct hostent *hp = NULL;

  if(NULL == (hp = gethostbyname(name))){
    fprintf(stderr, "unknown host %s\n", name);
    return -1;
  }

  sin->sin_family = hp->h_addrtype;
  memcpy(&sin->sin_addr, hp->h_addr, hp->h_length);

  return 0;
}

int net_Server_t::Listen(int port, const char * mask, const char * allow)
{
  struct sockaddr_in sin;
  struct sockaddr_in mask_sin;
  struct sockaddr_in allow_sin;
  int ns, len;

  struct hostent *hp = NULL;

  if(gethost(mask, &mask_sin)){
    goto error;
  }

  if(gethost(allow, &allow_sin)){
    goto error;
  }

  if(0 > (s = socket(AF_INET, SOCK_STREAM, 0))){
    fprintf(stderr, "socket: errno=%d\n", errno);
    goto error;
  }

  memset(&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons(port);
  sin.sin_addr.s_addr = INADDR_ANY;


  if(0 > bind(s, (struct sockaddr *)&sin, sizeof(sin))){
    fprintf(stderr, "bind: errno=%d\n", errno);
#ifndef WIN32
    shutdown(s, SHUT_RDWR);
#endif
    close(s);
    s = -1;
    goto error;
  }
  if(0 > listen(s, 1)){
    fprintf(stderr, "listen: errno=%d\n", errno);
#ifndef WIN32
    shutdown(s, SHUT_RDWR);
#endif
    close(s);
    s = -1;
    goto error;
  }

  // create thread pool
  int i;
  threadPool = new threadPool_t[maxConnections];
  for (i=0; i<maxConnections; i++) {
    threadPool[i].socket = -1;
#ifdef WIN32
    threadPool[i].sema = thrd_SemaCreate(0, 2, 0);
#else
    threadPool[i].sema = thrd_SemaCreate(0, 0, 0);
#endif
    threadPool[i].quit = 0;
    threadPool[i].server = 0;
  }

  len = sizeof(sin);
  while(0 <= (ns = accept(s, (struct sockaddr *)&sin, (socklen_t *)&len))){
    printf("connection request from %x:%d (socket %d)\n",
	   sin.sin_addr.s_addr, ntohs(sin.sin_port), ns);

    // TODO check localhost address need to goes through ntohs
    // not important for llrnet project, but important for portability
    // of this library.
    if (!(sin.sin_addr.s_addr == 0x100007f || 
	  (sin.sin_addr.s_addr & mask_sin.sin_addr.s_addr) == 
	  allow_sin.sin_addr.s_addr)) {

      printf(" --> address rejected \n");
      goto disconnect;
    }

  retry:
    // find an available thread slot
    // note : if the pool gets big we could maintain a linked list
    // of free slots, but it's not really worth it for small pools
    for (i=0; i<maxConnections; i++)
      if (threadPool[i].socket < 0) {
	// create the thread if necessary
	if (!threadPool[i].server) {
	  threadPool[i].server = this;
	  if(thrd_create_thread(threadPool[i].id, 
				connection_thread, threadPool + i) < 0) {
	    // this should not really happen unless we run out of 
	    // system resource
	    fprintf(stderr, "thrd_create_thread failed\n");
	    threadPool[i].server = 0;
	    continue;
	  }
	}
	break;
      }

    if (i < maxConnections) {
      threadPool[i].socket = ns;

      thrd_SemaSignal(threadPool[i].sema);
    } else {
      thrd_sleep(1);
      goto retry;

      printf(" --> connection refused, maximum (%d) connections reached\n",
	     maxConnections);
    disconnect:
#ifndef WIN32
      if (shutdown(ns, SHUT_RDWR))
	fprintf(stderr, "error while closing socket\n");
#endif
      close(ns);
    }
  }

  if(0 > ns){
    fprintf(stderr, "accept: errno=%d\n", errno);
  } else {
#ifndef WIN32
    shutdown(s, SHUT_RDWR);
#endif
    close(s);
    s = -1;
  }

  // shutdown thread pool
  for (i=0; i<maxConnections && threadPool[i].server; i++) {
    threadPool[i].quit = 1;
    thrd_SemaSignal(threadPool[i].sema);
  }

  // wait all worker threads have exited
  for (i=0; i<maxConnections && threadPool[i].server; i++) {
    while (threadPool[i].quit) {
      fprintf(stderr, "waiting thread #%d exits\n", i);
      thrd_sleep(1);
    }
  }

  delete[] threadPool;

  delete this;

  return 0;

 error:
  delete this;
  return -1;
}

int net_Server_t::OnConnect(int s)
{
#if 0
  int n, r;
  char buf[BUFSIZ];

  while(0 < (n = pfs_Read(buf, sizeof(buf), s)))
    if(n != (r = pfs_Write(buf, n, s))){
      fprintf(stderr, "send %d->%d\n", n, r);
      if(0 > r){
	fprintf(stderr, "send: errno=%d\n", errno);
	break;
      }
    }
  if(0 > n)
    fprintf(stderr, "recv: errno=%d\n", errno);
  pfs_CloseFile(s);
#endif

  return 0;
}



#ifdef WIN32
//#ifdef MFCGUI
extern "C" {
  //#endif
void OutputStr (char *buf);
  //#ifdef MFCGUI
}
//#endif
#endif

int net_Connect(const char * server_name, int port, int nonblock)
{
  struct sockaddr_in sin;
  int i, s, ret;

  //printf("Trying to connect to '%s' port %d\n", server_name, port);

#if defined(WIN32x)
  u_int temp;
  if((u_int)-1 != (temp = inet_addr(server_name))){
    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = temp;
  }
  else
#endif
  {
    struct hostent *hp = NULL;

    if(gethost(server_name, &sin)){
// #ifdef WIN32
//       char s[256];
//       sprintf(s, "unknown host %s\n", server_name);
//       OutputStr(s);
// #endif
      return -1;
    }
  }
  sin.sin_port = htons(port);

  for(i = 0; i < 5; i++){
    if(0 > (s = socket(AF_INET, SOCK_STREAM, 0))){
      fprintf(stderr, "socket() failed (errno=%d)\n", errno);
      return -1;
    }

    //#ifdef WIN32
    if (nonblock) {
      u_long mode = 1;
      ioctl(s, FIONBIO, &mode);

      connect(s, (struct sockaddr *)&sin,
	      sizeof(sin));
	goto ok;
    }
    //#endif

    if(0 == (ret = connect(s, (struct sockaddr *)&sin,
			   sizeof(sin))))
      goto ok;

// #ifdef WIN32
//     {
//     char s[256];
//     sprintf(s, "connect returns %d, errno = %d\n", ret, errno);
//     OutputStr(s);
//     }
// #endif

    close(s);

#ifndef WIN32
    if(errno != ECONNREFUSED){
      fprintf(stderr, "connect() failed (errno=%d)\n", errno);
      return -1;
    }
#endif
  }
  //fprintf(stderr, "too many refused\n");
  return -1;

 ok:
  //#ifdef WIN32
//   if (nonblock) {
//     u_long mode = 0;
//     ioctl(s, FIONBIO, &mode);
//   }
  //#endif

  return s;
}

int net_Close(int s)
{
#ifndef WIN32
  if (shutdown(s, SHUT_RDWR))
    fprintf(stderr, "shutdown: errno %d\n", errno);
#endif
  return close(s);
}

int net_Recv(int s, void * buffer, int length)
{
#ifdef WIN32xxx
  DWORD len;
  DWORD flags = 0;
  WSABUF buffers[] = { length, (char *)buffer };
  int res =  WSARecv(s, buffers, 1, &len, &flags, 0, 0);
  return !res ? (int)len : -1;
#else
  return recv(s, (char *) buffer, length, 0);
#endif
}

int net_Send(int s, const void * buffer, int length)
{
  return send(s, (const char *) buffer, length, 0);
}

int net_NonBlock(int s, int m)
{
  u_long mode = m;
  return ioctl(s, FIONBIO, &mode);
}
