/*
 * LLRNet - network part of LLR
 *
 * (C) 2004-2005 Vincent Penne
 *
 * Released under GNU LIBRARY GENERAL PUBLIC LICENSE
 * (See file LICENSE that must be included with this software)
 *
 */

/*
 * Thread, mutexes and semaphores support.
 */

#include "threads.h"

#ifdef WIN32

/* --- HEADERS --- */

#include <windows.h>
#include <string.h>

/* ---  CODE   --- */


int thrd_SemaCreate(int iniCount, int maxCount, void * userdata)
{
  HANDLE r = CreateSemaphore(NULL, iniCount, maxCount, NULL);

  return (int) (r == 0? -1 : (int)r);
}

int thrd_SemaDelete(int semId)
{
  int r = CloseHandle((HANDLE) semId);
  return r? 0 : -1;
}

int thrd_SemaWait(int semId)
{
  int r = WaitForSingleObject((HANDLE) semId, INFINITE);

  return (r == WAIT_ABANDONED || r == WAIT_OBJECT_0)? 0 : -1;
}

int thrd_SemaPoll(int semId)
{
  int r = WaitForSingleObject((HANDLE) semId, 0);

  return (r == WAIT_ABANDONED || r == WAIT_OBJECT_0)? 0 : -1;
}

int thrd_SemaSignal(int semId)
{
  int r = ReleaseSemaphore((HANDLE) semId, 1, NULL);

  return r? 0 : -1;
}
#else
/* --- HEADERS --- */
#include <semaphore.h>
#include <string.h>


#define MAX_SEMA 256

typedef struct sema_s {
  int next;
  
  sem_t sem;

  void * userdata;
} sema_t;


static sema_t semas[MAX_SEMA];
static int first_sema;
static Thrd_Mutex sema_mutex;

/* ---  CODE   --- */


static void Init()
{
  static int init;
  int i;

  if (init)
    return;

  init = 1;
  
  first_sema = 0;
  for (i=0; i<MAX_SEMA; i++)
    semas[i].next = i+1;
}


int thrd_SemaCreate(int iniCount, int maxCount, void * userdata)
{
  int i;

  sema_mutex.lock();
  Init();

  i = first_sema;
  if (i == MAX_SEMA) {
    sema_mutex.unlock();
    return -1;
  }

  //THRD_MSG("new semaphore %d\n", i);

  /* WARNING : not thread safe (really too bad :) )
     Fix this later ... FIXED */
  first_sema = semas[i].next;
  semas[i].next = -1;
  sema_mutex.unlock();

  semas[i].userdata = userdata;

  sem_init(&semas[i].sem, 0, maxCount);
  for ( ; iniCount<maxCount; iniCount++)
    sem_post(&semas[i].sem);

  return i;
}

int thrd_SemaDelete(int semId)
{
  int tmp;

  if (semId < 0 || semId >= MAX_SEMA || semas[semId].next != -1)
    return -1;

  sem_destroy(&semas[semId].sem);

  /* WARNING : not thread safe (really too bad :) )
     Fix this later ... FIXED */
  sema_mutex.lock();
  tmp = first_sema;
  first_sema = semId;
  semas[semId].next = tmp;
  sema_mutex.unlock();

  return 0;
}

int thrd_SemaWait(int semId)
{
  if (semId < 0 || semId >= MAX_SEMA || semas[semId].next != -1)
    return -1;

  sem_wait(&semas[semId].sem);

  return 0;
}

int thrd_SemaPoll(int semId)
{
  int res;

  if (semId < 0 || semId >= MAX_SEMA || semas[semId].next != -1)
    return -1;
  //  return PollSema(semId);

  return sem_trywait(&semas[semId].sem);
}

int thrd_SemaSignal(int semId)
{
  if (semId < 0 || semId >= MAX_SEMA || semas[semId].next != -1)
    return -1;

  sem_post(&semas[semId].sem);

  return 0;
}

#endif
