/*
 * LLRNet - network part of LLR
 *
 * (C) 2004-2005 Vincent Penne
 *
 * Released under GNU LIBRARY GENERAL PUBLIC LICENSE
 * (See file LICENSE that must be included with this software)
 *
 */

/*
 * Thread, mutexes and semaphores support.
 */


/* borrowed FROM fltk2 */
// Simple threads support functions.  These should not really be part of
// fltk, but without them we cannot write portable demo programs.  These
// should be useful for other programs.

// You must not include this file on systems that don't support either
// pthreads or WIN32 threads.  This is because all functions are defined
// inline here for maximum speed.

// You MUST call Fl::lock() before calling create_thread!  This is
// totally necessary, as calling Fl::unlock() without Fl::lock() having
// been called first will crash!  Fl::lock/unlock uses it's own internal
// implementation of mutex.

#ifndef Thrd_Threads_H
#define Thrd_Threads_H

#ifndef WIN32
// pthreads:

#include <pthread.h>
#include <assert.h>

typedef pthread_t Thrd_Thread;

#include <unistd.h>
static void thrd_sleep(int s) { sleep(s); }

static int thrd_create_thread(Thrd_Thread& t, void *(*f) (void *), void* p,
			      int stacksize = 64*1024)
{
  pthread_attr_t attr;
  pthread_attr_init(&attr);
  pthread_attr_setstacksize(&attr, stacksize);
  return pthread_create((pthread_t*)&t, &attr, f, p);
}

// Linux supports recursive locks, use them directly, with some cheating:
#ifdef PTHREAD_MUTEX_RECURSIVE_NP
//PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP

extern pthread_mutexattr_t Thrd_Mutex_attrib;

class Thrd_Mutex {
  friend class Thrd_SignalMutex;
  pthread_mutex_t mutex;
  Thrd_Mutex(const Thrd_Mutex&);
  Thrd_Mutex& operator=(const Thrd_Mutex&);
public:
  Thrd_Mutex() {pthread_mutex_init(&mutex, &Thrd_Mutex_attrib);}
  void lock() {pthread_mutex_lock(&mutex);}
  void unlock() {pthread_mutex_unlock(&mutex);}
  ~Thrd_Mutex() {pthread_mutex_destroy(&mutex);}
};

class Thrd_SignalMutex : public Thrd_Mutex {
  pthread_cond_t cond;
public:
  Thrd_SignalMutex() : Thrd_Mutex() {pthread_cond_init(&cond, 0);}
  void signal() {pthread_cond_broadcast(&cond);}
  void wait() {
#if 1
    assert("not implemented !");
#else
    int save_counter = mutex.m_count; mutex.m_count = 1;
    pthread_cond_wait(&cond, &mutex);
    mutex.m_count = save_counter;
#endif
  }
};

#else // standard pthread mutexes need a bit of work to be recursive:

class Thrd_Mutex {
  friend class Thrd_SignalMutex;
  pthread_mutex_t mutex;
  pthread_t owner;
  int counter;
  Thrd_Mutex(const Thrd_Mutex&);
  Thrd_Mutex& operator=(const Thrd_Mutex&);
public:
  Thrd_Mutex() : counter(0) {pthread_mutex_init(&mutex, 0);}
  void lock() {
    if (!counter || owner != pthread_self()) {
      pthread_mutex_lock(&mutex); owner = pthread_self();
    }
    counter++;
  }
  void unlock() {if (!--counter) pthread_mutex_unlock(&mutex);}
  ~Thrd_Mutex() {pthread_mutex_destroy(&mutex);}
};

class Thrd_SignalMutex : public Thrd_Mutex {
  pthread_cond_t cond;
public:
  Thrd_SignalMutex() : Thrd_Mutex() {pthread_cond_init(&cond, 0);}
  void signal() {pthread_cond_broadcast(&cond);}
  void wait() {
    int save_counter = counter; counter = 0;
    pthread_cond_wait(&cond, &mutex);
    counter = save_counter;
    owner = pthread_self();
  }
};

#endif

#else
// WIN32:

#include <windows.h>
#include <process.h>

typedef unsigned long Thrd_Thread;

static int thrd_create_thread(Thrd_Thread& t, void *(*f) (void *), void* p,
			      int stacksize = 64*1024) {
  return t = (Thrd_Thread)_beginthread((void( __cdecl * )( void * ))f, 0, p);
}

static void thrd_sleep(int s) { Sleep(s*1000); }

class Thrd_Mutex {
  friend class Thrd_SignalMutex;
  CRITICAL_SECTION cs;
  Thrd_Mutex(const Thrd_Mutex&);
  Thrd_Mutex& operator=(const Thrd_Mutex&);
public:
  Thrd_Mutex() {InitializeCriticalSection(&cs);}
  void lock() {EnterCriticalSection(&cs);}
  void unlock() {LeaveCriticalSection(&cs);}
  ~Thrd_Mutex() {DeleteCriticalSection(&cs);}
};

class Thrd_SignalMutex : public Thrd_Mutex {
  HANDLE event;
public:
  Thrd_SignalMutex() : Thrd_Mutex() {event = CreateEvent(0, FALSE, FALSE, 0);}
  void signal() {SetEvent(event);}
  void wait() {
    // int save_counter = cs.count; cs.count = 1;
    // the following three calls should be atomic, sigh...
    LeaveCriticalSection(&cs);
    WaitForSingleObject(event, INFINITE);
    EnterCriticalSection(&cs);
    // cs.count = save_counter;
  }
};

#endif

// Portable semaphores by me
typedef int thrd_Sema_t;

int thrd_SemaCreate(int iniCount, int maxCount, void * userdata);

int thrd_SemaDelete(int semId);

int thrd_SemaWait(int semId);

int thrd_SemaPoll(int semId);

int thrd_SemaSignal(int semId);


#endif
