/*----------------------------------------------------------------------
| This file contains routines and global variables that are common for
| all operating systems the program has been ported to.  It is included
| in one of the source code files of each port.  See Llr.h for the
| common #defines and common routine definitions.
+---------------------------------------------------------------------*/

#define	FFTTRYMAX 0
 
#define CHECK_IF_ANY_ERROR(X,J,N,K) \
/* If the sum of the output values is an error (such as infinity) */\
/* then raise an error. */\
\
		if (gw_test_illegal_sumout ()) {\
			sprintf (buf, ERRMSG0L, J, N, ERRMSG1A);\
			OutputBoth (buf);\
			goto error;\
		}\
\
/* Check that the sum of the input numbers squared is approximately */\
/* equal to the sum of unfft results.  Since this check may not */\
/* be perfect, check for identical results after a restart. */\
\
		if (gw_test_mismatched_sums ()) {\
			static unsigned long last_bit[10] = {0};\
			static double last_suminp[10] = {0.0};\
			static double last_sumout[10] = {0.0};\
			double suminp, sumout;\
			suminp = gwsuminp (X);\
			sumout = gwsumout (X);\
			if (J == last_bit[K] &&\
			    suminp == last_suminp[K] &&\
			    sumout == last_sumout[K]) {\
				writeResults (ERROK);\
				saving = 1;\
			} else {\
				char	msg[80]/*, sic[80], soc[80]*/;\
/*				sprintf (sic, "%.16g", suminp)*/;\
/*				sprintf (soc, "%.16g", sumout)*/;\
/*				if (strcmp(sic, soc)) {*/\
				    sprintf (msg, ERRMSG1B, suminp, sumout);\
				    sprintf (buf, ERRMSG0L, J, N, msg);\
				    if (nextffttaken >= FFTTRYMAX)	OutputBoth (buf);\
				    last_bit[K] = J;\
				    last_suminp[K] = suminp;\
				    last_sumout[K] = sumout;\
				    goto trynextfft;\
/*				}*/\
			}\
		}\
\
/* Check for excessive roundoff error  */\
\
		if (echk && MAXERR > 0.40) {\
			static unsigned long last_bit[10] = {0};\
			static double last_maxerr[10] = {0.0};\
			if (J == last_bit[K] &&\
			    MAXERR == last_maxerr[K]) {\
				writeResults (ERROK);\
				saving = 1;\
			} else {\
				char	msg[80];\
				sprintf (msg, ERRMSG1C, MAXERR);\
				sprintf (buf, ERRMSG0L, J, N, msg);\
				OutputBoth (buf);\
				last_bit[K] = J;\
				last_maxerr[K] = MAXERR;\
				goto error;\
			}\
		}\
\
		if (ERRCHK) {\
			if (MAXERR < reallyminerr && J > 30)\
				reallyminerr = MAXERR;\
			if (MAXERR > reallymaxerr)\
				reallymaxerr = MAXERR;\
		} 

char JUNK[]="Copyright 1996-2002 Just For Fun Software, All rights reserved";
char	INI_FILE[80] = {0};
char	RESFILE[80] = {0};
char	LOGFILE[80] = {0};
char	EXTENSION[8] = {0};

int volatile ERRCHK = 0;
unsigned int PRIORITY = 1;
unsigned int CPU_AFFINITY = 99;
EXTERNC unsigned int volatile CPU_TYPE = 0;
unsigned long volatile ITER_OUTPUT = 0;
unsigned long volatile ITER_OUTPUT_RES = 99999999;
unsigned long volatile DISK_WRITE_TIME = 30;
int	TWO_BACKUP_FILES = 1;
int	RUN_ON_BATTERY = 1;
int	TRAY_ICON = TRUE;
int	HIDE_ICON = FALSE;
unsigned int PRECISION = 2;
int	CUMULATIVE_TIMING = 0;
int	HIGH_RES_TIMER = 0; 


/* PRP and LLR global variables */

#define	sgkbufsize 20000

giant	N = NULL;		/* Number being tested */

unsigned long Nlen;		/* Bit length of number being LLRed or PRPed */

unsigned long klen = 0;	/* Number of bits of k multiplier */
char testbuffer[256] = {0};
unsigned int skiptest = 0;
unsigned int testkynea = 0;
unsigned int testcarol = 0;
unsigned int verbose = 0;

int is_valid_double(double);
int genProthBase(giant, unsigned long);

/* Utility output routines */

void LineFeed ()
{
	OutputStr ("\n");
}

void OutputNum (
	unsigned long num)
{
	char	buf[20];
	sprintf (buf, "%lu", num);
	OutputStr (buf);
}

/* Sleep five minutes before restarting */

char ERROK[] = "Disregard last error.  Result is reproducible and thus not a hardware problem.\n";
char ERRMSG0L[] = "Iter: %ld/%ld, %s";
char ERRMSG0[] = "Bit: %ld/%ld, %s"; 
char ERRMSG1A[] = "ERROR: ILLEGAL SUMOUT\n";
char ERRMSG1B[] = "ERROR: SUM(INPUTS) != SUM(OUTPUTS), %.16g != %.16g\n";
char ERRMSG1C[] = "ERROR: ROUND OFF (%.10g) > 0.40\n";
char ERRMSG2[] = "Possible hardware failure, consult the readme file.\n";
char ERRMSG3[] = "Continuing from last save file.\n";
char ERRMSG4[] = "Waiting five minutes before restarting.\n";
char ERRMSG5[] = "fftlen seems to be too small, using next fftlen...\n";
char WRITEFILEERR[] = "Error writing intermediate file: %s\n";

int SleepFive ()
{
	int	i;

	OutputStr (ERRMSG4);
	BlinkIcon (10);			/* Blink icon for 10 seconds */
	Sleep (10000);
	ChangeIcon (IDLE_ICON);		/* Idle icon for rest of 5 minutes */
	for (i = 0; i < 290; i++) {
		Sleep (1000);
		if (escapeCheck ()) return (FALSE);
	}
	ChangeIcon (WORKING_ICON);	/* And back to the working icon */
	return (TRUE);
}

/* Truncate a percentage to the requested number of digits. */
/* Truncating prevents 99.5% from showing up as 100% complete. */

double trunc_percent (
	double	percent)
{
	if (percent > 100.0) percent = 100.0;
	percent -= 0.5 * pow (10.0, - (double) PRECISION);
	if (percent < 0.0) return (0.0);
	return (percent);
}

 
//  Test if a string contains only valid digits. 
 
int isDigitString(char *s) { 
    while (*s) { 
	if (!isdigit(*s)) return (FALSE); 
	s++; 
    } 
    return (TRUE); 
} 
 
/* Routines used to time code chunks */

double timers[10];			/* Up to ten separate timers */

void clear_timers () {
	int	i;
	for (i = 0; i < 10; i++) timers[i] = 0.0;
}

void clear_timer (
	int	i)
{
	timers[i] = 0.0;
}

void start_timer ( 
	int	i) 
{ 
	if (HIGH_RES_TIMER) { 
		timers[i] -= getHighResTimer (); 
	} else { 
		struct _timeb timeval; 
		_ftime (&timeval); 
		timers[i] -= (double) timeval.time * 1000.0 + timeval.millitm; 
	} 
} 
 
void end_timer ( 
	int	i) 
{ 
	if (HIGH_RES_TIMER) { 
		timers[i] += getHighResTimer (); 
	} else { 
		struct _timeb timeval; 
		_ftime (&timeval); 
		timers[i] += (double) timeval.time * 1000.0 + timeval.millitm; 
	} 
} 
 
void divide_timer (
	int	i,
	int	j)
{
	timers[i] = timers[i] / j;
}

double timer_value ( 
	int	i) 
{ 
	if (HIGH_RES_TIMER) 
		return (timers[i] / getHighResTimerFrequency ()); 
	else 
		return (timers[i] / 1000.0); 
} 
 
#define TIMER_NL	0x1
#define TIMER_CLR	0x2
#define TIMER_OPT_CLR	0x4

void print_timer (
	int	i,
	int	flags)
{ 
	char	buf[40]; 
	double	t; 
 
	t = timer_value (i); 
	if (t >= 1.0) 
		sprintf (buf, "%.3f sec.", t); 
	else 
		sprintf (buf, "%.3f ms.", t * 1000.0); 
	OutputStr (buf); 
	if (flags & TIMER_NL) LineFeed (); 
	if (flags & TIMER_CLR) timers[i] = 0.0; 
	if ((flags & TIMER_OPT_CLR) && !CUMULATIVE_TIMING) timers[i] = 0.0; 
} 
 
void write_timer (		// JP 17/11/01
	char* buf,
	int	i, 
	int	flags) 
{ 
	double	t; 
 
	t = timer_value (i); 
	if (t >= 1.0) { 
	    if (flags & TIMER_NL)
		sprintf (buf, "%.3f sec.\n", t); 
	    else
		sprintf (buf, "%.3f sec.", t); 
	}
	else { 
	    if (flags & TIMER_NL)
		sprintf (buf, "%.3f ms.\n", t * 1000.0); 
	    else
		sprintf (buf, "%.3f ms.", t * 1000.0); 
	}
 
	if (flags & TIMER_CLR) timers[i] = 0.0; 
	if ((flags & TIMER_OPT_CLR) && !CUMULATIVE_TIMING) timers[i] = 0.0; 
} 

/* Determine the CPU speed either empirically or by user overrides. */ 
/* getCpuType must be called prior to calling this routine. */ 
 
void getCpuSpeed (void) 
{ 
	int	temp; 
 
/* Guess the CPU speed using the RDTSC instruction */ 
 
	guessCpuSpeed (); 
 
/* Now let the user override the cpu speed from the local.ini file */ 
 
	if (IniGetInt (INI_FILE, "CpuOverride", 0)) { 
		temp = IniGetInt (INI_FILE, "CpuSpeed", 99); 
		if (temp != 99) CPU_SPEED = temp; 
	} 
 
/* Make sure the cpu speed is reasonable */ 
 
	if (CPU_SPEED > 50000) CPU_SPEED = 50000; 
	if (CPU_SPEED < 25) CPU_SPEED = 25; 
} 
 
/* Set the CPU flags based on the CPUID data.  Also, the */ 
/* advanced user can override our guesses. */ 
 
unsigned int SAVE_CPU_FLAGS;

void getCpuInfo (void) 
{ 
	int	temp; 
 
/* Get the CPU info using CPUID instruction */ 
 
	guessCpuType (); 
 
/* Let the user override the cpu flags from the local.ini file */ 
 
	temp = IniGetInt (INI_FILE, "CpuSupportsRDTSC", 99); 
	if (temp == 0) CPU_FLAGS &= ~CPU_RDTSC; 
	if (temp == 1) CPU_FLAGS |= CPU_RDTSC; 
	temp = IniGetInt (INI_FILE, "CpuSupportsCMOV", 99); 
	if (temp == 0) CPU_FLAGS &= ~CPU_CMOV; 
	if (temp == 1) CPU_FLAGS |= CPU_CMOV; 
	temp = IniGetInt (INI_FILE, "CpuSupportsPrefetch", 99); 
	if (temp == 0) CPU_FLAGS &= ~CPU_PREFETCH; 
	if (temp == 1) CPU_FLAGS |= CPU_PREFETCH; 
	temp = IniGetInt (INI_FILE, "CpuSupportsSSE", 99); 
	if (temp == 0) CPU_FLAGS &= ~CPU_SSE; 
	if (temp == 1) CPU_FLAGS |= CPU_SSE; 
	temp = IniGetInt (INI_FILE, "CpuSupportsSSE2", 99); 
	if (temp == 0) CPU_FLAGS &= ~CPU_SSE2; 
	if (temp == 1) CPU_FLAGS |= CPU_SSE2; 
 
SAVE_CPU_FLAGS = CPU_FLAGS;		// Duplicate these values

/* Now get the CPU speed */ 
 
	getCpuSpeed (); 
} 
 
/* Format a long or very long textual cpu description */ 
 
void getCpuDescription ( 
	char	*buf,			/* A 512 byte buffer */ 
	int	long_desc)		/* True for a very long description */ 
{ 
 
/* Recalculate the CPU speed in case speed step has changed the original */ 
/* settings. */ 
 
	getCpuSpeed (); 
 
/* Now format a pretty CPU description */ 
 
	sprintf (buf, "%s\nCPU speed: %.2f MHz\n", CPU_BRAND, CPU_SPEED); 
	if (CPU_FLAGS) { 
		strcat (buf, "CPU features: "); 
		if (CPU_FLAGS & CPU_RDTSC) strcat (buf, "RDTSC, "); 
		if (CPU_FLAGS & CPU_CMOV) strcat (buf, "CMOV, "); 
		if (CPU_FLAGS & CPU_PREFETCH) strcat (buf, "PREFETCH, "); 
		if (CPU_FLAGS & CPU_MMX) strcat (buf, "MMX, "); 
		if (CPU_FLAGS & CPU_SSE) strcat (buf, "SSE, "); 
		if (CPU_FLAGS & CPU_SSE2) strcat (buf, "SSE2, "); 
		strcpy (buf + strlen (buf) - 2, "\n"); 
	} 
	strcat (buf, "L1 cache size: "); 
	if (CPU_L1_CACHE_SIZE < 0) strcat (buf, "unknown\n"); 
	else sprintf (buf + strlen (buf), "%d KB\n", CPU_L1_CACHE_SIZE); 
	strcat (buf, "L2 cache size: "); 
	if (CPU_L2_CACHE_SIZE < 0) strcat (buf, "unknown\n"); 
	else sprintf (buf + strlen (buf), "%d KB\n", CPU_L2_CACHE_SIZE); 
	if (! long_desc) return; 
	strcat (buf, "L1 cache line size: "); 
	if (CPU_L1_CACHE_LINE_SIZE < 0) strcat (buf, "unknown\n"); 
	else sprintf (buf+strlen(buf), "%d bytes\n", CPU_L1_CACHE_LINE_SIZE); 
	strcat (buf, "L2 cache line size: "); 
	if (CPU_L2_CACHE_LINE_SIZE < 0) strcat (buf, "unknown\n"); 
	else sprintf (buf+strlen(buf), "%d bytes\n", CPU_L2_CACHE_LINE_SIZE); 
	if (CPU_L1_DATA_TLBS > 0) 
		sprintf (buf + strlen (buf), "L1 TLBS: %d\n", CPU_L1_DATA_TLBS); 
	if (CPU_L2_DATA_TLBS > 0) 
		sprintf (buf + strlen (buf), "%sTLBS: %d\n", 
			 CPU_L1_DATA_TLBS > 0 ? "L2 " : "", 
			 CPU_L2_DATA_TLBS); 
} 
 
/* Determine if a number is prime */

int isPrime (
	unsigned long p)
{
	unsigned long i;
	for (i = 2; i * i <= p; i = (i + 1) | 1)
		if (p % i == 0) return (FALSE);
	return (TRUE);
}

/* Determine the names of the INI files */

void nameIniFiles (
	int	named_ini_files)
{
	char	buf[120];

	if (named_ini_files < 0) {
		strcpy (INI_FILE, "llr.ini");
		strcpy (RESFILE, "lresults.txt");
		strcpy (LOGFILE, "lprime.log");
		strcpy (EXTENSION, "");
	} else {
		sprintf (INI_FILE, "llr%04d.ini", named_ini_files);
		sprintf (RESFILE, "lresu%04d.txt", named_ini_files);
		sprintf (LOGFILE, "lprim%04d.log", named_ini_files);
		sprintf (EXTENSION, ".%03d", named_ini_files);
	}

/* Let the user rename these files and pick a different working directory */

	IniGetString (INI_FILE, "WorkingDir", buf, sizeof(buf), NULL);
	IniGetString (INI_FILE, "results.txt", RESFILE, 80, RESFILE);
	IniGetString (INI_FILE, "prime.log", LOGFILE, 80, LOGFILE);
	IniGetString (INI_FILE, "prime.ini", INI_FILE, 80, INI_FILE);
	if (buf[0]) {
		_chdir (buf);
		IniFileOpen (INI_FILE, 0);
	}
}

/* Read the INI files */

void readIniFiles () 
{ 
	int	temp; 
 
	getCpuInfo (); 
 
	PRECISION = (unsigned int) IniGetInt (INI_FILE, "PercentPrecision", 2); 
	if (PRECISION > 6) PRECISION = 6; 
 
	ITER_OUTPUT = IniGetInt (INI_FILE, "OutputIterations", 10000); 
	if (ITER_OUTPUT <= 0) ITER_OUTPUT = 1; 
	ITER_OUTPUT_RES = IniGetInt (INI_FILE, "ResultsFileIterations", 
				     99999999); 
	if (ITER_OUTPUT_RES < 1000) ITER_OUTPUT_RES = 1000; 
	DISK_WRITE_TIME = IniGetInt (INI_FILE, "DiskWriteTime", 30); 
	TWO_BACKUP_FILES = (int) IniGetInt (INI_FILE, "TwoBackupFiles", 1); 
	RUN_ON_BATTERY = (int) IniGetInt (INI_FILE, "RunOnBattery", 1); 
 
	temp = (int) IniGetInt (INI_FILE, "ErrorCheck", 0); 
	ERRCHK = (temp != 0); 
	PRIORITY = (unsigned int) IniGetInt (INI_FILE, "Priority", 1); 
	CPU_AFFINITY = (unsigned int) IniGetInt (INI_FILE, "Affinity", 99); 
	HIDE_ICON = (int) IniGetInt (INI_FILE, "HideIcon", 0); 
	TRAY_ICON = (int) IniGetInt (INI_FILE, "TrayIcon", 1); 
 
/* Guess the CPU type if it isn't known.  Otherwise, validate it. */ 
 
	getCpuInfo (); 
 
/* Other oddball options */ 
 
	CUMULATIVE_TIMING = IniGetInt (INI_FILE, "CumulativeTiming", 0); 
	HIGH_RES_TIMER = isHighResTimerAvailable (); 
} 
 
/*----------------------------------------------------------------------
| Portable routines to read and write ini files!  NOTE:  These only
| work if you open no more than 5 ini files.  Also you must not
| change the working directory at any time during program execution.
+---------------------------------------------------------------------*/

struct IniLine {
	char	*keyword;
	char	*value;
	int	active;
};
struct IniCache {
	char	*filename;
	int	immediate_writes;
	int	dirty;
	unsigned int num_lines;
	unsigned int array_size;
	struct IniLine **lines;
};

void growIniLineArray (
	struct IniCache *p)
{
	struct IniLine **newlines;

	if (p->num_lines != p->array_size) return;

	newlines = (struct IniLine **)
		malloc ((p->num_lines + 100) * sizeof (struct IniLine **));
	if (p->num_lines) {
		memcpy (newlines, p->lines, p->num_lines * sizeof (struct IniLine *));
		free (p->lines);
	}
	p->lines = newlines;
	p->array_size = p->num_lines + 100;
}

struct IniCache *openIniFile (
	char	*filename,
	int	forced_read)
{
static	struct IniCache *cache[10] = {0};
	struct IniCache *p;
	FILE	*fd;
	unsigned int i;
	char	line[80];
	char	*val;

/* See if file is cached */

	for (i = 0; i < 10; i++) {
		p = cache[i];
		if (p == NULL) {
			p = (struct IniCache *) malloc (sizeof (struct IniCache));
			p->filename = (char *) malloc (strlen (filename) + 1);
			strcpy (p->filename, filename);
			p->immediate_writes = 1;
			p->dirty = 0;
			p->num_lines = 0;
			p->array_size = 0;
			p->lines = NULL;
			forced_read = 1;
			cache[i] = p;
			break;
		}
		if (strcmp (filename, p->filename) == 0)
			break;
	}

/* Skip reading the ini file if appropriate */

	if (!forced_read) return (p);
	if (p->dirty) return (p);

/* Free the data if we've already read some in */

	for (i = 0; i < p->num_lines; i++) {
		free (p->lines[i]->keyword);
		free (p->lines[i]->value);
		free (p->lines[i]);
	}
	p->num_lines = 0;

/* Read the IniFile */
	
	fd = fopen (filename, "r");
	if (fd == NULL) return (p);

	while (fgets (line, 80, fd)) {
		if (line[strlen(line)-1] == '\n') line[strlen(line)-1] = 0;
		if (line[0] == 0) continue;
		if (line[strlen(line)-1] == '\r') line[strlen(line)-1] = 0;
		if (line[0] == 0) continue;

		val = strchr (line, '=');
		if (val == NULL) {
			char	buf[130];
			sprintf (buf, "Illegal line in INI file: %s\n", line);
			OutputSomewhere (buf);
			continue;
		}
		*val++ = 0;

		growIniLineArray (p);
		
/* Allocate and fill in a new line structure */

		i = p->num_lines++;
		p->lines[i] = (struct IniLine *) malloc (sizeof (struct IniLine));
		p->lines[i]->keyword = (char *) malloc (strlen (line) + 1);
		p->lines[i]->value = (char *) malloc (strlen (val) + 1);
		p->lines[i]->active = TRUE;
		strcpy (p->lines[i]->keyword, line);
		strcpy (p->lines[i]->value, val);
	}
	fclose (fd);

	return (p);
}

void writeIniFile (
	struct IniCache *p)
{
	int	fd;
	unsigned int j;
	char	buf[100];

/* Delay writing the file unless this INI file is written */
/* to immediately */

	if (!p->immediate_writes) {
		p->dirty = 1;
		return;
	}

/* Create and write out the INI file */

	fd = _open (p->filename, _O_CREAT | _O_TRUNC | _O_WRONLY | _O_TEXT, 0666);
	if (fd < 0) return;
	for (j = 0; j < p->num_lines; j++) {
		strcpy (buf, p->lines[j]->keyword);
		strcat (buf, "=");
		strcat (buf, p->lines[j]->value);
		strcat (buf, "\n");
		_write (fd, buf, strlen (buf));
	}
	p->dirty = 0;
	_close (fd);
}

void truncated_strcpy (
	char	*buf,
	unsigned int bufsize,
	char	*val)
{
	if (strlen (val) >= bufsize) {
		memcpy (buf, val, bufsize-1);
		buf[bufsize-1] = 0;
	} else {
		strcpy (buf, val);
	}
}

void IniGetString (
	char	*filename,
	char	*keyword,
	char	*val,
	unsigned int val_bufsize,
	char	*default_val)
{
	struct IniCache *p;
	unsigned int i;

/* Open ini file */

	p = openIniFile (filename, 1);

/* Look for the keyword */

	for (i = 0; ; i++) {
		if (i == p->num_lines) {
			if (default_val == NULL) {
				val[0] = 0;
			} else {
				truncated_strcpy (val, val_bufsize, default_val);
			}
			return;
		}
		if (p->lines[i]->active &&
		    stricmp (keyword, p->lines[i]->keyword) == 0) break;
	}

/* Copy info from the line structure to the user buffers */

	truncated_strcpy (val, val_bufsize, p->lines[i]->value);
}

long IniGetInt (
	char	*filename,
	char	*keyword,
	long	default_val)
{
	char	buf[20], defval[20];
	sprintf (defval, "%ld", default_val);
	IniGetString (filename, keyword, buf, 20, defval);
	return (atol (buf));
}

void IniWriteString (
	char	*filename,
	char	*keyword,
	char	*val)
{
	struct IniCache *p;
	unsigned int i, j;

/* Open ini file */

	p = openIniFile (filename, 1);

/* Look for the keyword */

	for (i = 0; ; i++) {
		if (i == p->num_lines ||
		    stricmp (p->lines[i]->keyword, "Time") == 0) {

/* Ignore request if we are deleting line */

			if (val == NULL) return;

/* Make sure the line array has room for the new line */

			growIniLineArray (p);

/* Shuffle entries down to make room for this entry */

			for (j = p->num_lines; j > i; j--)
				p->lines[j] = p->lines[j-1];

/* Allocate and fill in a new line structure */

			p->lines[i] = (struct IniLine *) malloc (sizeof (struct IniLine));
			p->lines[i]->keyword = (char *) malloc (strlen (keyword) + 1);
			strcpy (p->lines[i]->keyword, keyword);
			p->lines[i]->value = NULL;
			p->num_lines++;
			break;
		}
		if (p->lines[i]->active &&
		    stricmp (keyword, p->lines[i]->keyword) == 0) {
			if (val != NULL && strcmp (val, p->lines[i]->value) == 0) return;
			break;
		}
	}

/* Delete the line if requested */

	if (val == NULL) {
		IniDeleteLine (filename, i+1);
		return;
	}

/* Replace the value associated with the keyword */

	free (p->lines[i]->value);
	p->lines[i]->value = (char *) malloc (strlen (val) + 1);
	strcpy (p->lines[i]->value, val);

/* Write the INI file back to disk */

	writeIniFile (p);
}

void IniWriteInt (
	char	*filename,
	char	*keyword,
	long	val)
{
	char	buf[20];
	sprintf (buf, "%ld", val);
	IniWriteString (filename, keyword, buf);
}

void IniFileOpen (
	char	*filename,
	int	immediate_writes)
{
	struct IniCache *p;
	p = openIniFile (filename, 1);
	p->immediate_writes = immediate_writes;
}

void IniFileClose (
	char	*filename)
{
	struct IniCache *p;
	p = openIniFile (filename, 0);
	if (p->dirty) {
		p->immediate_writes = 1;
		writeIniFile (p);
		p->immediate_writes = 0;
	}
}

int IniFileWritable (
	char	*filename)
{
	struct IniCache *p;
	int	fd;
	unsigned int j;
	char	buf[100];

/* Create and write out the INI file */

	p = openIniFile (filename, 0);
	fd = _open (p->filename, _O_CREAT | _O_TRUNC | _O_WRONLY | _O_TEXT, 0666);
	if (fd < 0) return (FALSE);
	for (j = 0; j < p->num_lines; j++) {
		strcpy (buf, p->lines[j]->keyword);
		strcat (buf, "=");
		strcat (buf, p->lines[j]->value);
		strcat (buf, "\n");
		if (_write (fd, buf, strlen (buf)) != (int) strlen (buf)) {
			_close (fd);
			return (FALSE);
		}
	}
	if (p->num_lines == 0) {
		if (_write (fd, "DummyLine=XXX\n", 14) != 14) {
			_close (fd);
			return (FALSE);
		}
		p->dirty = 1;
	}
	_close (fd);
	return (TRUE);
}

unsigned int IniGetNumLines (
	char	*filename)
{
	struct IniCache *p;
	p = openIniFile (filename, 0);
	return (p->num_lines);
}

void IniGetLineAsString (
	char	*filename,
	unsigned int line,
	char	*keyword,
	unsigned int keyword_bufsize,
	char	*val,
	unsigned int val_bufsize)
{
	struct IniCache *p;

/* Open ini file */

	p = openIniFile (filename, 0);

/* Copy info from the line structure to the user buffers */

	truncated_strcpy (keyword, keyword_bufsize, p->lines[line-1]->keyword);
	truncated_strcpy (val, val_bufsize, p->lines[line-1]->value);
}

void IniGetLineAsInt (
	char	*filename,
	unsigned int line,
	char	*keyword,
	unsigned int keyword_bufsize,
	long	*val)
{
	char	buf[20];
	IniGetLineAsString (filename, line, keyword, keyword_bufsize, buf, 20);
	*val = atol (buf);
}

void IniReplaceLineAsString (
	char	*filename,
	unsigned int line,
	char	*keyword,
	char	*val)
{
	IniDeleteLine (filename, line);
	IniInsertLineAsString (filename, line, keyword, val);
}

void IniReplaceLineAsInt (
	char	*filename,
	unsigned int line,
	char	*keyword,
	long	val)
{
	char	buf[20];
	sprintf (buf, "%ld", val);
	IniReplaceLineAsString (filename, line, keyword, buf);
}

void IniInsertLineAsString (
	char	*filename,
	unsigned int line,
	char	*keyword,
	char	*val)
{
	struct IniCache *p;
	unsigned int i;

/* Open ini file, do not reread it as that could change the line numbers! */

	p = openIniFile (filename, 0);

/* Adjust line number if it doesn't make sense */

	if (line == 0) line = 1;
	if (line > p->num_lines+1) line = p->num_lines+1;

/* Make sure the line array has room for the new line */

	growIniLineArray (p);

/* Shuffle lines down in the array to make room for the new line */

	for (i = p->num_lines; i >= line; i--) p->lines[i] = p->lines[i-1];
	p->num_lines++;

/* Allocate and fill in a new line structure */

	p->lines[line-1] = (struct IniLine *) malloc (sizeof (struct IniLine));
	p->lines[line-1]->keyword = (char *) malloc (strlen (keyword) + 1);
	p->lines[line-1]->value = (char *) malloc (strlen (val) + 1);
	p->lines[line-1]->active = TRUE;
	strcpy (p->lines[line-1]->keyword, keyword);
	strcpy (p->lines[line-1]->value, val);

/* Write the INI file back to disk */

	writeIniFile (p);
}

void IniInsertLineAsInt (
	char	*filename,
	unsigned int line,
	char	*keyword,
	long	val)
{
	char	buf[20];
	sprintf (buf, "%ld", val);
	IniInsertLineAsString (filename, line, keyword, buf);
}

void IniAppendLineAsString (
	char	*filename,
	char	*keyword,
	char	*val)
{
	struct IniCache *p;
	p = openIniFile (filename, 0);
	IniInsertLineAsString (filename, p->num_lines+1, keyword, val);
}

void IniAppendLineAsInt (
	char	*filename,
	char	*keyword,
	long	val)
{
	char	buf[20];
	sprintf (buf, "%ld", val);
	IniAppendLineAsString (filename, keyword, buf);
}

void IniDeleteLine (
	char	*filename,
	unsigned int line)
{
	struct IniCache *p;
	unsigned int i;

/* Open ini file, do not reread it as that could change the line numbers! */

	p = openIniFile (filename, 0);
	if (line == 0 || line > p->num_lines) return;

/* Free the data associated with the given line */

	free (p->lines[line-1]->keyword);
	free (p->lines[line-1]->value);
	free (p->lines[line-1]);

/* Delete the line from the lines array */

	for (i = line; i < p->num_lines; i++) p->lines[i-1] = p->lines[i];
	p->num_lines--;

/* Write the INI file back to disk */

	writeIniFile (p);
}

void IniDeleteAllLines (
	char	*filename)
{
	struct IniCache *p;
	unsigned int i;

/* Open ini file! */

	p = openIniFile (filename, 0);

/* Free the data associated with the given line */

	for (i = 0; i < p->num_lines; i++) {
		free (p->lines[i]->keyword);
		free (p->lines[i]->value);
		free (p->lines[i]);
	}
	p->num_lines = 0;

/* Write the INI file back to disk */

	writeIniFile (p);
}

/* Output string to screen or results file */

void OutputSomewhere (
	char	*buf)
{
	if (NO_GUI) writeResults (buf);
	else OutputStr (buf);
}

/* Output string to both the screen and results file */

void OutputBoth (
	char	*buf)
{
	OutputStr (buf);
	writeResults (buf);
}

/* Output message to screen and prime.log file */

void LogMsg (
	char	*str)
{
	int	fd;
	unsigned long filelen;
static	time_t	last_time = 0;
	time_t	this_time;

/* Output it to the screen */

	OutputStr (str);

/* Open the log file and position to the end */

	fd = _open (LOGFILE, _O_TEXT | _O_RDWR | _O_CREAT, 0666);
	if (fd < 0) {
		OutputStr ("Unable to open log file.\n");
		return;
	}
	filelen = _lseek (fd, 0L, SEEK_END);

/* Output to the log file only if it hasn't grown too big */

	if (filelen < 250000) {

/* If it has been at least 5 minutes since the last time stamp */
/* was output, then output a new timestamp */

		time (&this_time);
		if (this_time - last_time > 300) {
			char	buf[48];
			last_time = this_time;
			buf[0] = '[';
			strcpy (buf+1, ctime (&this_time));
			sprintf (buf+25, " - ver %s]\n", VERSION);
			_write (fd, buf, strlen (buf));
		}

/* Output the message */

		_write (fd, str, strlen (str));
	}

/* Display message about full log file */
	
	else {
		char	*fullmsg = "Prime.log file full.  Please delete it.\n";
		OutputStr (fullmsg);
		if (filelen < 251000)
			_write (fd, fullmsg, strlen (fullmsg));
	}
	_close (fd);
}

int gmodi (unsigned long, giant);
void uldivg (unsigned long, giant);


/* Generate temporary file name */

void tempFileName ( 
	char	*buf, char c) 
{ 
	int remainder;
 
	remainder = gmodi(19999981, N);
	sprintf (buf, "%01c%07li", c, remainder % 10000000); 
} 
 
/* See if the given file exists */

int fileExists (
	char	*filename)
{
	int	fd;
	fd = _open (filename, _O_RDONLY | _O_BINARY);
	if (fd < 0) return (0);
	_close (fd);
	return (1);
}

/* Open the results file and write a line to the end of it. */

int writeResults (
	char	*msg)
{
static	time_t	last_time = 0;
	time_t	this_time;
	int	fd;

/* Open file, position to end */

	fd = _open (RESFILE, _O_TEXT | _O_RDWR | _O_CREAT | _O_APPEND, 0666);
	if (fd < 0) {
		LogMsg ("Error opening the results file.\n");
		return (FALSE);
	}

/* If it has been at least 5 minutes since the last time stamp */
/* was output, then output a new timestamp */

	time (&this_time);
	if (this_time - last_time > 300) {
		char	buf[32];
		last_time = this_time;
		buf[0] = '[';
		strcpy (buf+1, ctime (&this_time));
		buf[25] = ']';
		buf[26] = '\n';
		if (verbose)
			_write (fd, buf, 27);
	}

/* Output the message */

	if (_write (fd, msg, strlen (msg)) < 0) goto fail;
	_close (fd);
	return (TRUE);

/* On a write error, close file and return error flag */

fail:	_close (fd);
	return (FALSE);
}


/* Read and write intermediate results to a file */

int read_gwnum (
	int	fd,
	gwnum	g,
	long	*sum)
{
	giant	tmp;
	long	i, len, bytes;

	tmp = popg (((unsigned long)bit_length >> 5) + 10);
	if (_read (fd, &len, sizeof (long)) != sizeof (long)) return (FALSE);
	bytes = len * sizeof (long);
	if (_read (fd, tmp->n, bytes) != bytes) return (FALSE);
	tmp->sign = len;
	*sum += len;
	for (i = 0; i < len; i++) *sum += tmp->n[i];
	gianttogw (tmp, g);
	pushg (1);
	return (TRUE);
}

int write_gwnum (
	int	fd,
	gwnum	g,
	long	*sum)
{
	giant	tmp;
	long	i, len, bytes;

	tmp = popg (((unsigned long)bit_length >> 5) + 10);
	gwtogiant (g, tmp);
	len = tmp->sign;
	if (_write (fd, &len, sizeof (long)) != sizeof (long)) return (FALSE);
	bytes = len * sizeof (long);
	if (_write (fd, tmp->n, bytes) != bytes) return (FALSE);
	*sum += len;
	for (i = 0; i < len; i++) *sum += tmp->n[i];
	pushg (1);
	return (TRUE);
}

int read_long (
	int	fd,
	unsigned long *val,
	long	*sum)
{
	if (_read (fd, val, sizeof (long)) != sizeof (long)) return (FALSE);
	*sum += *val;
	return (TRUE);
}

int write_long (
	int	fd,
	unsigned long val,
	long	*sum)
{
	if (_write (fd, &val, sizeof (long)) != sizeof (long)) return (FALSE);
	*sum += val;
	return (TRUE);
}

int writeToFile (
	char	*filename,
	unsigned long j,
	gwnum	x,
	gwnum	y)
{
	char	newfilename[16];
	int	fd;
	unsigned long magicnum, version;
	long	sum = 0;

/* If we are allowed to create multiple intermediate files, then */
/* write to a file called yNNNNNNN. */

	strcpy (newfilename, filename);
	if (TWO_BACKUP_FILES) newfilename[0] = 'y';

/* Create the intermediate file */

	fd = _open (newfilename, _O_BINARY|_O_WRONLY|_O_TRUNC|_O_CREAT, 0666);
	if (fd < 0) return (FALSE);

/* Write the file header. */

	magicnum = 0x9f2b3cd4;
	if (_write (fd, &magicnum, sizeof (long)) != sizeof (long))
		goto writeerr;
	version = 1;
	if (_write (fd, &version, sizeof (long)) != sizeof (long))
		goto writeerr;

/* Write the file data */

	if (! write_long (fd, j, &sum)) goto writeerr;

/* Write the data values */

	if (! write_gwnum (fd, x, &sum)) goto writeerr;
	if (y != NULL && ! write_gwnum (fd, y, &sum)) goto writeerr; 

/* Write the checksum */

	if (_write (fd, &sum, sizeof (long)) != sizeof (long)) goto writeerr;
	_commit (fd);
	_close (fd);

/* Now rename the intermediate files */

	if (TWO_BACKUP_FILES) {
		_unlink (filename);
		rename (newfilename, filename);
	}
	return (TRUE);

/* An error occured.  Close and delete the current file. */

writeerr:
	_close (fd);
	_unlink (newfilename);
	return (FALSE);
}

int readFromFile (
	char	*filename,
	unsigned long *j,
	gwnum	x,
	gwnum	y)
{
	int	fd;
	unsigned long magicnum, version;
	long	sum = 0, i;

/* Open the intermediate file */

	fd = _open (filename, _O_BINARY | _O_RDONLY);
	if (fd < 0) goto error;

/* Read the file header */

	if (_read (fd, &magicnum, sizeof (long)) != sizeof (long))
		goto readerr;
	if (magicnum != 0x9f2b3cd4) goto readerr;

	if (_read (fd, &version, sizeof (long)) != sizeof (long)) goto readerr;
	if (version != 1 && version != 2) goto readerr;

/* Read the file data */

	if (! read_long (fd, j, &sum)) goto readerr;

/* Read the values */

	if (! read_gwnum (fd, x, &sum)) goto readerr;
	if (y != NULL && ! read_gwnum (fd, y, &sum)) goto readerr; 

/* Read and compare the checksum */

	if (_read (fd, &i, sizeof (long)) != sizeof (long)) goto readerr;
	if (i != sum) goto readerr;
	_close (fd);
	return (TRUE);

/* An error occured.  Delete the current intermediate file. */
/* Set stage to -1 to indicate an error. */

readerr:
	OutputStr ("Error reading LLR or PRP save file.\n");
	_close (fd);
error:
	_unlink (filename);
	return (FALSE);
}

void	trace(int n) {			// Debugging tool...
	char buf[100];
	sprintf(buf, "OK until number %d\n", n);
	OutputBoth (buf); 	
}

char res64[17]; /* VP : This variable has been made global */

/* Test for a probable prime -- gwsetup has already been called. */

int commonPRP (
	unsigned long a,
	int	*res, char *str)
{
	unsigned long bit, iters;
	gwnum	x;
	giant	tmp;
	char	filename[20], buf[sgkbufsize+256], fft_desc[100], oldres64[17];
	long	write_time = DISK_WRITE_TIME * 60;
	int	echk, saving, stopping;
	time_t	start_time, current_time;
	double	reallyminerr = 1.0;
	double	reallymaxerr = 0.0;

/* Init, subtract 1 from N to compute a^(N-1) mod N */

	iaddg (-1, N);
	Nlen = bitlen (N);
	*res = TRUE;		/* Assume it is a probable prime */

/* Init filename */

	tempFileName (filename, 'z');

/* Get the current time */

	clear_timers ();
	start_timer (0);
	start_timer (1);
	time (&start_time);

/* Allocate memory */

	x = gwalloc ();

/* Optionally resume from save file and output a message */
/* indicating we are resuming a test */

	if (fileExists (filename) && readFromFile (filename, &bit, x, NULL)) {
		char	fmt_mask[80];
		double	pct;
		pct = trunc_percent (bit * 100.0 / Nlen);
		sprintf (fmt_mask,
			 "Resuming probable prime test of %%s at bit %%ld [%%.%df%%%%]\n",
			 PRECISION);
		sprintf (buf, fmt_mask, str, bit, pct);
		OutputStr (buf);
	}

/* Otherwise, output a message indicating we are starting test */

	else {
		sprintf (buf, "Starting probable prime test of %s\n", str);
		OutputStr (buf);
		bit = 1;
		dbltogw ((double) a, x);
	}

/* Output a message about the FFT length */

	gwfft_description (fft_desc);
	sprintf (buf, "Using %s\n\n", fft_desc);
	OutputStr (buf);
	ReplaceableLine (1);	/* Remember where replaceable line is */

/* Init the title */

	if (strlen (str) < 40)
		title (str);
	else
		title("Running in Big k mode...");

/* Do the PRP test */

//for (int i =0;i < 20; i++) set_fft_value (x, i, 0);
//set_fft_value (x, 16, 200000);
//set_fft_value (x, 16, 0);
//#define CHECK_ITER
#ifdef CHECK_ITER
{giant t1, t2;
t1 = popg ((Nlen >> 4) + 3);
t2 = popg ((Nlen >> 4) + 3);
gwtogiant (x, t1);
#endif
	gwsetmulbyconst (a);
	iters = 0;
	while (bit < Nlen) {

/* Error check the last 50 iterations, before writing an */
/* intermediate file (either user-requested stop or a */
/* 30 minute interval expired), and every 128th iteration. */

		stopping = stopCheck ();
		echk = stopping || ERRCHK || bit+50 >= Nlen;
		if ((bit & 127) == 0) {
			echk = 1;
			time (&current_time);
			saving = (current_time - start_time > write_time);
		} else
			saving = 0;

/* Process this bit */

		gwstartnextfft (!stopping && !saving && bit+26 < Nlen);
#ifdef CHECK_ITER
squareg (t1);
if (bitval (N, Nlen-bit-1)) imulg (a, t1);
specialmodg (t1);
gwstartnextfft (0);
echk=1;
#endif
		if (bitval (N, Nlen-bit-1)) {
			gwsetnormroutine (0, echk, 1);
		} else {
			gwsetnormroutine (0, echk, 0);
		}
		if (bit+25 < Nlen) gwsquare (x);
		else gwsquare_carefully (x);

#ifdef CHECK_ITER
gwtogiant (x, t2);
if (gcompg (t1, t2) != 0)
OutputStr ("Iteration failed.\n");
//if (bit == 100) bit = Nlen;
#endif

/* If the sum of the output values is an error (such as infinity) */
/* then raise an error. */

		if (gw_test_illegal_sumout ()) {
			sprintf (buf, ERRMSG0, bit, Nlen, ERRMSG1A);
			OutputBoth (buf);
			goto error;
		}

/* Check that the sum of the input numbers squared is approximately */
/* equal to the sum of unfft results.  Since this check may not */
/* be perfect, check for identical results after a restart. */

		if (gw_test_mismatched_sums ()) {
			static unsigned long last_bit = 0;
			static double last_suminp = 0.0;
			static double last_sumout = 0.0;
			double suminp, sumout;
			suminp = gwsuminp (x);
			sumout = gwsumout (x);
			if (bit == last_bit &&
			    suminp == last_suminp &&
			    sumout == last_sumout) {
				writeResults (ERROK);
				saving = 1;
			} else {
				char	msg[80];
				sprintf (msg, ERRMSG1B, suminp, sumout);
				sprintf (buf, ERRMSG0, bit, Nlen, msg);
				OutputBoth (buf);
				last_bit = bit;
				last_suminp = suminp;
				last_sumout = sumout;
				goto error;
			}
		}

/* Check for excessive roundoff error  */

		if (echk && MAXERR > 0.40) {
			static unsigned long last_bit = 0;
			static double last_maxerr = 0.0;
			if (bit == last_bit &&
			    MAXERR == last_maxerr) {
				writeResults (ERROK);
				saving = 1;
			} else {
				char	msg[80];
				sprintf (msg, ERRMSG1C, MAXERR);
				sprintf (buf, ERRMSG0, bit, Nlen, msg);
				OutputBoth (buf);
				last_bit = bit;
				last_maxerr = MAXERR;
				goto error;
			}
		}

/* Keep track of maximum and minimum round off error */

		if (ERRCHK) {
			if (MAXERR < reallyminerr && bit > 30)
				reallyminerr = MAXERR;
			if (MAXERR > reallymaxerr)
				reallymaxerr = MAXERR;
		}

/* That iteration succeeded, bump counters */

		bit++;
		iters++;

/* Print a message every so often */

		if (bit % ITER_OUTPUT == 0) {
			char	fmt_mask[80];
			double	pct;
			pct = trunc_percent (bit * 100.0 / Nlen);
			sprintf (fmt_mask, "%%.%df%%%% of %%ld", PRECISION);
			sprintf (buf, fmt_mask, pct, Nlen);
			title (buf);
			ReplaceableLine (2);	/* Replace line */
			sprintf (fmt_mask,
				 "%%s, bit: %%ld / %%ld [%%.%df%%%%]",
				 PRECISION);
			sprintf (buf, fmt_mask, str, bit, Nlen, pct);
			OutputStr (buf);
			if (ERRCHK && bit > 30) {
				OutputStr (".  Round off: ");
				sprintf (buf, "%10.10f", reallyminerr);
				OutputStr (buf);
				sprintf (buf, " to %10.10f", reallymaxerr);
				OutputStr (buf);
			}
			end_timer (0);
			if (CUMULATIVE_TIMING) {
				OutputStr (".  Time thusfar: ");
			} else {
				OutputStr (".  Time per bit: ");
				divide_timer (0, iters);
				iters = 0;
			}
			print_timer (0, TIMER_NL | TIMER_OPT_CLR);
			start_timer (0);
		}

/* Print a results file message every so often */

		if (bit % ITER_OUTPUT_RES == 0 || (NO_GUI && stopping)) {
			sprintf (buf, "Bit %ld / %ld\n", bit, Nlen);
			writeResults (buf);
		}

/* Write results to a file every DISK_WRITE_TIME minutes */
/* On error, retry in 10 minutes (it could be a temporary */
/* disk-full situation) */

		if (saving || stopping) {
			write_time = DISK_WRITE_TIME * 60;
			if (! writeToFile (filename, bit, x, NULL)) {
				sprintf (buf, WRITEFILEERR, filename);
				OutputBoth (buf);
				if (write_time > 600) write_time = 600;
			}
			time (&start_time);

/* If an escape key was hit, write out the results and return */

			if (stopping) {
				gwfree (x);
				gwdone ();
				*res = FALSE;		// To avoid credit message !
				return (FALSE);
			}
		}
	}
#ifdef CHECK_ITER
pushg(2);}
#endif

/* See if we've found a probable prime.  If not, format a 64-bit residue. */
/* Old versions of PRP used a non-standard 64-bit residue, computing */
/* 3^N-3 mod N rather than the more standard 3^(N-1) mod N.  Since */
/* some projects recorded these non-standard residues, output that */
/* residue too.  Note that some really old versions printed out the */
/* 32-bit chunks of the non-standard residue in reverse order. */

	tmp = popg ((Nlen >> 5) + 3);
	gwtogiant (x, tmp);
	if (!isone (tmp)) {
		*res = FALSE;	/* Not a prime */
		if (abs(tmp->sign) < 2)	// make a 32 bit residue correct !!
			tmp->n[1] = 0;
		sprintf (res64, "%08lX%08lX", tmp->n[1], tmp->n[0]);
		imulg (3, tmp); specialmodg (tmp); ulsubg (3, tmp);
		sprintf (oldres64, "%08lX%08lX", tmp->n[1], tmp->n[0]);
	}
	pushg (1);
	gwfree (x);

/* Print results.  Do not change the format of this line as Jim Fougeron of */
/* PFGW fame automates his QA scripts by parsing this line. */

	if (*res)
		sprintf (buf, "%s is a probable prime.\n", str);
	else if (IniGetInt (INI_FILE, "OldRes64", 0))
		sprintf (buf, "%s is not prime.  RES64: %s.  OLD64: %s\n", str, res64, oldres64);
	else
		sprintf (buf, "%s is not prime.  RES64: %s\n", str, res64);

/* Update the output file */

/*	if ((*res && IniGetInt (INI_FILE, "OutputPrimes", 1)) ||
	    (!*res && IniGetInt (INI_FILE, "OutputComposites", 1)))
		writeResults (buf); */

/* Output the final timings */

	end_timer (1);
	sprintf (buf+strlen(buf)-1, "  Time: ");
	ReplaceableLine (2);	/* Replace line */
	write_timer (buf+strlen(buf), 1, TIMER_CLR | TIMER_NL); 
	OutputBoth (buf);

/* Cleanup and return */

	gwdone ();
	_unlink (filename);
	return (TRUE);

/* An error occured, sleep, then try restarting at last save point. */

error:
	gwfree (x);
	gwdone ();

/* Output a message saying we are restarting */

	OutputBoth (ERRMSG2);
	OutputBoth (ERRMSG3);
	*res = FALSE;					// To avoid credit mesage...

/* Sleep five minutes before restarting */

	if (! SleepFive ()) return (FALSE);

/* Restart */

	return (-1);
}

/* Test if K*2^N+C is a probable prime. */

int fastIsPRP (
	double	k,			/* K in k*b^n+c */
	unsigned long b,		/* B in k*b^n+c */
	unsigned long n,		/* N in k*b^n+c */
	signed long c,			/* C in k*b^n+c */
	int	*res)
{
	int	retval;
//	double	bits;

//k=10223.0;n=6565829;c=1;

/* Compute the number we are testing */ // Already done here !

/*	bits = log ((double) b) / log (2.0) * (double) n;
	N = newgiant (((unsigned long) bits >> 4) + 8);
	ultog (b, N);
	power (N, n);
	dblmulg (k, N);
	iaddg (c, N);

/* Setup the assembly code. */

	do {
		gwsetup (k, b, n, c, 0);

/* Do the PRP test */

		retval = commonPRP (3, res, gwmodulo_as_string ());
	} while (retval == -1);

/* Clean up and return */

//	free (N);
	return (retval);
}


/* Test if N is a probable prime.  The number N can be of ANY form. */

int slowIsPRP (
	char	*str,		/* string representation of N */
	int	*res)
{
	int	retval;

/* Setup the gwnum code */

	do {
		gwsetup_general_mod (N, 0);
//		strcpy (GWSTRING_REP, str);

/* Do the PRP test */

		retval = commonPRP (3, res, str);
	} while (retval == -1);
	return (retval);
}


/* Test if a small N is a probable prime. */
/* Compute 3^(N-1) mod N */

int isProbablePrime (void)
{
	int	retval;
	giant	x;

	if (isone (N)) return (FALSE);
	x = newgiant (N->sign + 8);
	itog (3, x);
	powermodg (x, N, N);
	iaddg (-3, x);
	retval = isZero (x);
	free (x);
	return (retval);
}


/* Multiply numbers using a slower method that will have reduced */
/* round-off error on non-random input data.  Caller must make sure the */
/* input numbers has not been partially or fully FFTed. */

void gwmul_carefully (
	gwnum	s, gwnum t)		/* Source, destination */
{
	gwnum	tmp1, tmp2, tmp3;
	double	saved_addin_value;

/* Generate a random number, if we have't already done so */

	if (GW_RANDOM == NULL) {
		GW_RANDOM = gwalloc ();
		gw_random_number (GW_RANDOM);
	}

/* Save and clear the addin value */

	saved_addin_value = ADDIN_VALUE;
	ADDIN_VALUE = 0.0;

/* Now do the squaring using three multiplies and adds */

	tmp1 = gwalloc ();
	tmp2 = gwalloc ();
	tmp3 = gwalloc ();
	gwstartnextfft (0);			/* Disable POSTFFT */
	gwadd3 (s, GW_RANDOM, tmp1);		/* Compute s+random */
	gwadd3 (t, GW_RANDOM, tmp3);		/* Compute t+random */
	gwfft (GW_RANDOM, tmp2);
	gwaddquick (s, t);
	gwfftmul (tmp2, t);			/* Compute (s+t)*random */
	gwfftfftmul (tmp2, tmp2, tmp2);		/* Compute random^2 */
	ADDIN_VALUE = saved_addin_value;	/* Restore the addin value */
	gwmul (tmp1, tmp3);			/* Compute (s+random)*(t+random) */
	gwsubquick (tmp2, tmp3);		/* Calc s^t from 3 results */
	gwsub3 (tmp3, t, t);

/* Free memory and return */

	gwfree (tmp1);
	gwfree (tmp2);
	gwfree (tmp3);
}

/*
 	Primality testing of k*2^n-1 numbers with the Lucas Lehmer Riesel
	algorithm, using the gwnums for fast multiplications and squarings.
	Second attempt for a full IBDWT version, using Colin Percival method improved
	by George Woltman.
	Jean Penn, May 2004.
*/

unsigned long getnextfftlength (void) {
	unsigned long *jmptab;

	if (!FFTLEN) return (0);
	jmptab = INFT[0];
	for (jmptab = jmptab+14; *jmptab; jmptab++);
	jmptab++;

	if (jmptab[0]) {
		return (jmptab[1]);
	}

	return (0);

}

int isLLRP_p ( 
	giant gk, 
	char *sgk,
	unsigned long n, 
	int	*res) 
{ 
	unsigned long len, iters, index, word2; 
	unsigned long p, fftlen, fftlmers, j, k; 
	unsigned long mask, last, bit; 
	long	vindex, retval;
	long	nextffttaken = 0;
	gwnum	x, y, gwn2; 
	giant	tmp; 
	char	filename[20], buf[sgkbufsize+256], str[sgkbufsize+256], str2[sgkbufsize+256]; 
	long	write_time = DISK_WRITE_TIME * 60; 
	int	echk, saving, stopping, v1, first = 1, inc = -1; 
	time_t	start_time, current_time; 
	double	reallyminerr = 1.0; 
	double	reallymaxerr = 0.0; 
	double dk, gwn2w;

	sprintf (str, "%s*2^%lu%c1", sgk, n, '-');

//	gk must be odd for the LLR test, so, adjust gk and n if necessary.

	if (!(testkynea || testcarol)) {
		dk = atof(sgk);				// k value as a double.
		sprintf (str2, "%s*2^%lu%c1", sgk, n, '-');

		while (bitval(gk, 0) == 0) {
			gshiftright (1, gk);	// update k as a giant
			dk /= 2.0;				// and as a double
			n++;
		}
	}
	else {
		dk = 0.0;
		if (testkynea)
			sprintf (str2, "%s*2^%lu%c1 = (2^%d+1)^2 - 2", sgk, n, '-', n-1);
		else
			sprintf (str2, "%s*2^%lu%c1 = (2^%d-1)^2 - 2", sgk, n, '-', n-1);
	}

	Nlen = len = bitlen (N); 
	klen = bitlen(gk);

	if (N->sign == 1) {		// N is a small number, so we make a simple test...
		if (*res = isPrime (N->n[0]))
			sprintf (buf, "%s = %lu is prime! (trial divisions)\n", str2, N->n[0]); 
		else
			sprintf (buf, "%s = %lu is  not prime. (trial divisions)\n", str2, N->n[0]); 
		OutputBoth (buf);
		return (TRUE); 
	}

	if (klen > n) {
	    sprintf(buf, "%s > 2^%lu, so we can only do a PRP test for %s.\n", sgk, n, str);
	    OutputBoth(buf);
		if (is_valid_double (dk))
			retval = fastIsPRP (dk, 2, n, -1, res);
		else if (Nlen < 50) {
			*res = isProbablePrime();
			if (*res)
				sprintf (buf, "%s is a probable prime.\n", str);
			else
				sprintf (buf, "%s is not prime.\n", str);
			OutputBoth (buf);
			retval = TRUE;
		}
		else
			retval = slowIsPRP (str, res);
		if (*res)
			OutputBoth ("Please credit George Woltman's PRP for this result!\n");
		return retval;
	}

	if(abs(gk->sign) == 1) {	// k is a "small" integer
	    k = gk->n[0];
	}
	else {
		k = 0;					// to indicate that k is a big integer
	}
 

/* Get the current time */ 

restart: 
	vindex = 1;				// First attempt
	nextffttaken = 0;
	time (&start_time); 

/* Assume intermediate results of the length of N. */ 
/* Compute the fftlen for a Mersenne number of this size. */
	
	p = len; 

	fftlmers = gwmap_to_fftlen (1.0, 2, p, -1);

	fftlen = 0;			// for first attempt...

	*res = TRUE;		/* Assume it is prime */ 

	if (!(testkynea || testcarol) && klen < 48 && is_valid_double(dk))
		gwsetup (dk, 2, n, -1, fftlen); 
	else {
		gwsetup_general_mod (N, fftlen);
	}



	clear_timers (); 
	start_timer (0); 
	start_timer (1); 

	x = gwalloc (); 
	y = gwalloc ();
	gwn2 = gwalloc ();

	if (k != 1) {					// Non Mersenne
		bitaddr (n+1, &word2, &bit);
		gwn2w = *addr (gwn2, word2);
	}
	else {							// Mersenne
		word2 = 0;
		gwn2w = 2.0;
	}

	if (skiptest && klen <= 22 && gwn2w >= 0) {
		sprintf (buf, "No need to redo L. L. R. prime test of %s\n", str2); 
		OutputStr (buf); 
		gwfree (x); 
		gwfree (y);
		gwfree (gwn2);
		gwdone();
		*res = FALSE;
		end_timer (1); 
		return(TRUE);
	}

 	last = n-1;

 	gwsetnormroutine (0, ERRCHK, 0); 

/* Init filename */ 

	tempFileName (filename, 'z'); 
 
/* Optionally resume from save file and output a message */ 
/* indicating we are resuming a test */ 
 
	if (fileExists (filename) && readFromFile (filename, &j, x, NULL)) { 
		char	fmt_mask[80]; 
		double	pct; 
		pct = trunc_percent (j * 100.0 / n); 
		sprintf (fmt_mask, 
			 "Resuming test of %%s at iteration %%ld [%%.%df%%%%]\n", 
			 PRECISION); 
		sprintf (buf, fmt_mask, str2, j, pct); 
		OutputStr (buf); 
	} 
 

/* Otherwise, output a message indicating we are starting test, */ 
/* or resuming the computing of U0. */
 
	else { 
	    if (k==1) {
			if (!isPrime (n)) {
				sprintf (buf, "The Mersenne number %s is not prime because %d is not prime.\n", str, n); 
				OutputBoth (buf); 
				gwfree (x); 
				gwfree (y);
				gwfree (gwn2);
				gwdone();
				*res = FALSE;
				end_timer (1); 
				return(TRUE);
			}
		sprintf (buf, 
	      "Prime95 or Mprime are much better to test this Mersenne number !!\n");
		if (verbose)
			OutputBoth(buf);
		else
			OutputStr(buf);
		v1 = 4;
		dbltogw ((double) v1, x); 
		goto MERSENNE;
	    }
	    filename[0] = 'u';

	    if ((v1 = gen_v1(gk, n, 0, vindex)) < 0) {
			if (v1 == -1)
				sprintf (buf, "Cannot compute V1 to test %s...\nThis is surprising, please, let me know that!!\nMy E-mail is jpenne@wanadoo.fr\n", str2);
			else
				sprintf (buf, "%s has a small factor : %d !!\n", str2, abs(v1));
			OutputBoth (buf); 
			gwfree (x); 
			gwfree (y);
			gwfree (gwn2);
			gwdone();
			*res = FALSE;
			end_timer (1); 
			return(TRUE);
	    }
	    if (fileExists (filename) && readFromFile (filename, &j, x, y)) { 
			char	fmt_mask[80]; 
			double	pct; 
			pct = trunc_percent (100.0 - j * 100.0 / klen); 
			sprintf (fmt_mask, 
			 "Resuming test of %%s (computing U0) at iteration %%ld [%%.%df%%%%]\n", 
			 PRECISION); 
			sprintf (buf, fmt_mask, str2, klen - j, pct); 
			OutputStr (buf); 
			ReplaceableLine (1);	/* Remember where replacable line is */ 
	    } 
	    else {
			if (!skiptest) {
				sprintf (buf, "Starting Lucas Lehmer Riesel prime test of %s\n", str2);
				OutputStr (buf); 
			}
			else {
				sprintf (buf, "Redoing Lucas Lehmer Riesel prime test of %s\n", str2); 
				OutputBoth (buf); 
			}
			if (!GENERAL_MOD)
				if (!ZERO_PADDED_FFT)
					if (!RATIONAL_FFT)
						sprintf (buf, "Using Irrational Base DWT : Mersenne fftlen = %d, Used fftlen = %d\n", fftlmers, FFTLEN);
					else
						sprintf (buf, "Using Rational Base DWT : Mersenne fftlen = %d, Used fftlen = %d\n", fftlmers, FFTLEN);
				else
					if (!RATIONAL_FFT)
						sprintf (buf, "Using Zero Padded IBDWT : Mersenne fftlen = %d, Used fftlen = %d\n", fftlmers, FFTLEN);
					else
						sprintf (buf, "Using Zero Padded RBDWT : Mersenne fftlen = %d, Used fftlen = %d\n", fftlmers, FFTLEN);
			else
				if (!RATIONAL_FFT)
					sprintf (buf, "Using General Mode (Irrational Base) : Mersenne fftlen = %d, Used fftlen = %d\n", fftlmers, FFTLEN);
				else
					sprintf (buf, "Using General Mode (Rational Base) : Mersenne fftlen = %d, Used fftlen = %d\n", fftlmers, FFTLEN);
			if (verbose)
				OutputBoth(buf);
			else
				OutputStr(buf);

			sprintf (buf, "V1 = %d ; Computing U0...\n", v1);
			OutputStr (buf); 

			ReplaceableLine (1);	/* Remember where replacable line is */ 

			dbltogw ((double) v1, x); 
			dbltogw ((double) v1, y); 
			gwsetaddin (-2);
//			gwsquare (y); 
			gwsquare_carefully (y); 
			CHECK_IF_ANY_ERROR(y, 1, klen, 0)
			j = klen - 2;
	    }
					/* Computing u0 (cf Hans Riesel) */
	    iters = 0; 
	    while (j>0) {
 
/* Process this iteration */ 
 
			mask = 1<<j;

			if (k)
				bit = (k&mask);
			else
				bit = bitval (gk, j);
 
			index = klen-j--;
			iters++;

/* Error check the last 50 iterations, before writing an */ 
/* intermediate file (either user-requested stop or a */ 
/* 30 minute interval expired), and every 128th iteration. */ 
 
			stopping = stopCheck (); 
			echk = stopping || ERRCHK || (j <= 50); 
			if ((index & 127) == 0) { 
				echk = 1; 
				time (&current_time); 
				saving = (current_time - start_time > write_time); 
			}
			else 
				saving = 0; 
			gwfft (x, x);
			gwfft (y, y);
			if (bit) {
				gwsetaddin (-v1);

//				gwsafemul (y, x);
				gwfftfftmul (y, x, x);
				CHECK_IF_ANY_ERROR(x, (index), klen, 1)
				gwsetaddin (-2);
//				gwsquare (y);
				gwfftfftmul (y, y, y);
				CHECK_IF_ANY_ERROR(y, (index), klen, 2)
			}
			else {
				gwsetaddin (-v1);
//				gwsafemul (x, y);
				gwfftfftmul (x, y, y);
				CHECK_IF_ANY_ERROR(y, (index), klen, 3)
				gwsetaddin (-2);
//				gwsquare (x);
				gwfftfftmul (x, x, x);
				CHECK_IF_ANY_ERROR(x, (index), klen, 4)
			}
 
/* Print a message every so often */ 
 
			if (index % ITER_OUTPUT == 0) { 
				char	fmt_mask[80]; 
				double	pct; 
				pct = trunc_percent (100.0 - j * 100.0 / klen); 
				sprintf (fmt_mask, "%%.%df%%%% of %%ld", PRECISION); 
				sprintf (buf, fmt_mask, pct, klen); 
				title (buf); 
				ReplaceableLine (2);	/* Replace line */ 
				sprintf (fmt_mask, 
				 "%%s, iteration : %%ld / %%ld [%%.%df%%%%]", 
				 PRECISION); 
				sprintf (buf, fmt_mask, str, index, klen, pct); 
				OutputStr (buf); 
				if (ERRCHK && index > 30) { 
					OutputStr (".  Round off: "); 
					sprintf (buf, "%10.10f", reallyminerr); 
					OutputStr (buf); 
					sprintf (buf, " to %10.10f", reallymaxerr); 
					OutputStr (buf); 
				} 
				end_timer (0); 
				if (CUMULATIVE_TIMING) { 
					OutputStr (".  Time thusfar : "); 
				} 
				else { 
					OutputStr (".  Time per iteration : "); 
					divide_timer (0, iters); 
					iters = 0; 
				} 
				print_timer (0, TIMER_NL | TIMER_OPT_CLR); 
				start_timer (0); 
			} 
 
/* Print a results file message every so often */ 
 
			if (index % ITER_OUTPUT_RES == 0 || (NO_GUI && stopping)) { 
				sprintf (buf, "Iteration %ld / %ld\n", index, klen); 
				writeResults (buf); 
			} 
 
/* Write results to a file every DISK_WRITE_TIME minutes */ 
/* On error, retry in 10 minutes (it could be a temporary */ 
/* disk-full situation) */ 
 
			if (saving || stopping) { 
				write_time = DISK_WRITE_TIME * 60; 
				if (! writeToFile (filename, j, x, y)) { 
					sprintf (buf, WRITEFILEERR, filename); 
					OutputBoth (buf); 
					if (write_time > 600) write_time = 600; 
				} 
				time (&start_time); 
 
/* If an escape key was hit, write out the results and return */ 
 
				if (stopping) {
					gwfree (x); 
					gwfree (y);
					gwfree (gwn2);
					gwdone();
					return (FALSE); 
				}
			} 
	    }

		gwfft (x, x);
		gwfft (y, y);
		gwsetaddin (-v1);
//	    gwmul (y, x);
		gwfftfftmul (y, x, x);
	    CHECK_IF_ANY_ERROR(x, klen, klen, 5)
		ReplaceableLine (2);	/* Replace line */ 
		sprintf (buf, "V1 = %d ; Computing U0...done.\n", v1);
		if (verbose)
			OutputBoth (buf); 
		else
			OutputStr(buf);
							/* End of x = u0 computing */
	    filename[0] = 'z';	/* restore filename which was modified... */
MERSENNE:
	    j = 1;
	    sprintf (buf, "Starting Lucas-Lehmer loop...\n"); 
	    OutputStr (buf); 
	} 
 
	ReplaceableLine (1);	/* Remember where replacable line is */ 
 
/* Init the title */ 
 
	if (strlen (str) < 40)
		title(str);
	else
		title("Running in Big k mode...");
 
/* Do the Lucas Lehmer Riesel Prime test */ 
 
	iters = 0; 
	gwsetaddin (-2);

	while (j<last) { 
 
/* Error check the last 50 iterations, before writing an */ 
/* intermediate file (either user-requested stop or a */ 
/* 30 minute interval expired), and every 128th iteration. */ 

		stopping = stopCheck (); 
		echk = stopping || ERRCHK || (j >= last - 50); 
		if ((j & 127) == 0) { 
			echk = 1; 
			time (&current_time); 
			saving = (current_time - start_time > write_time); 
		} else 
			saving = 0; 


/* Process this iteration */ 

		gwstartnextfft (!stopping && !saving && j+26 < last); 

		if (j < last - 25) {
			gwsquare (x);
		}
		else {
			gwsquare_carefully(x);
		}
		CHECK_IF_ANY_ERROR(x, j, last, 6)
		j++; 
		iters++; 
 
 
/* Print a message every so often */ 
 
		if (j % ITER_OUTPUT == 0) { 
			char	fmt_mask[80]; 
			double	pct; 
			pct = trunc_percent (j * 100.0 / n); 
			sprintf (fmt_mask, "%%.%df%%%% of %%ld", PRECISION); 
			sprintf (buf, fmt_mask, pct, n); 
			title (buf); 
			ReplaceableLine (2);	/* Replace line */ 
			sprintf (fmt_mask, 
				 "%%s, iteration : %%ld / %%ld [%%.%df%%%%]", 
				 PRECISION); 
			sprintf (buf, fmt_mask, str, j, n, pct); 
			OutputStr (buf); 
			if (ERRCHK && j > 30) { 
				OutputStr (".  Round off: "); 
				sprintf (buf, "%10.10f", reallyminerr); 
				OutputStr (buf); 
				sprintf (buf, " to %10.10f", reallymaxerr); 
				OutputStr (buf); 
			} 
			end_timer (0); 
			if (CUMULATIVE_TIMING) { 
				OutputStr (".  Time thusfar : "); 
			} 
			else { 
				OutputStr (".  Time per iteration : "); 
				divide_timer (0, iters); 
				iters = 0; 
			} 
			print_timer (0, TIMER_NL | TIMER_OPT_CLR); 
			start_timer (0); 
		} 
 
/* Print a results file message every so often */ 
 
		if (j % ITER_OUTPUT_RES == 0 || (NO_GUI && stopping)) { 
			sprintf (buf, "Iteration %ld / %ld\n", j, n); 
			writeResults (buf); 
		} 
 
/* Write results to a file every DISK_WRITE_TIME minutes */ 
/* On error, retry in 10 minutes (it could be a temporary */ 
/* disk-full situation) */ 
 
		if (saving || stopping) { 
			write_time = DISK_WRITE_TIME * 60; 

			if (! writeToFile (filename, j, x, NULL)) { 
				sprintf (buf, WRITEFILEERR, filename); 
				OutputBoth (buf); 
				if (write_time > 600) write_time = 600; 
			} 
			time (&start_time); 

 
/* If an escape key was hit, write out the results and return */ 
 
			if (stopping) {
				gwfree (x); 
				gwfree (y);
				gwfree (gwn2);
				gwdone();
				return (FALSE); 
			}
		} 
	} 

	tmp =  popg (((unsigned long)bit_length >> 5) + 3); 

	gwtogiant (x, tmp); 
	if (!isZero (tmp)) { 
		*res = FALSE;	/* Not a prime */ 
		if (abs(tmp->sign) < 2)	// make a 32 bit residue correct !!
			tmp->n[1] = 0;
		sprintf (res64, "%08lX%08lX", tmp->n[1], tmp->n[0]); 
	} 
 
/* Cleanup */ 
 
	end_timer (1); 
	if (*res) 
		sprintf (buf, "%s is prime!\n", str2); 
	else
		sprintf (buf, "%s is not prime.  Res64: %s\n", str2, res64); 
	sprintf (buf+strlen(buf)-1, "  Time : "); 
	ReplaceableLine (2);	/* Replace line */ 
	write_timer (buf+strlen(buf), 1, TIMER_CLR | TIMER_NL); 
	OutputBoth (buf); 

	pushg (1); 
	gwfree (x); 
	gwfree (y);
	gwfree (gwn2);
	gwdone (); 
	filename[0] = 'u';
	_unlink (filename);
	filename[0] = 'z';
	_unlink (filename); 
	return (TRUE); 

trynextfft:	
 
/* An error occured, sleep, then try restarting at last save point. */ 

error:
	gwfree (x); 
	gwfree (y); 
	gwfree (gwn2);
	gwdone (); 

/* Output a message saying we are restarting */ 
 
	OutputBoth (ERRMSG2); 
	OutputBoth (ERRMSG3); 
 
/* Sleep five minutes before restarting */ 
 
	if (! SleepFive ()) return (FALSE); 
 
/* Restart */ 
 
	goto restart; 
} 

static unsigned long mask;

int isProthP_p ( 
	giant gk, 
	char *sgk,
	unsigned long n, 
	int	*res) 
{ 
	unsigned long len, iters; 
	unsigned long p, fftlen, fftlmers; 
	unsigned long bit; 
	long	a, retval;
	gwnum	x; 
	giant	tmp; 
	char	filename[20], buf[sgkbufsize+256], str[sgkbufsize+256], fft_desc[100]; 
	long	write_time = DISK_WRITE_TIME * 60; 
	int	echk, saving, stopping, inc = +1; 
	time_t	start_time, current_time; 
	double	reallyminerr = 1.0; 
	double	reallymaxerr = 0.0; 
	double dk;

	sprintf (str, "%s*2^%lu%c1", sgk, n, '+');

//	gk must be odd for the Proth test, so, adjust gk and n if necessary.

	dk = atof(sgk);				// k value as a double.

	while (bitval(gk, 0) == 0) {
	    gshiftright (1, gk);	// update k as a giant
		dk /= 2.0;				// and as a double
	    n++;
	}

/* Compute again the number we are testing */ 

//	gtog (gk, N); 
//	gshiftleft (n, N); 
//	iaddg (inc, N); 
	Nlen = len = bitlen (N); 
	klen = bitlen(gk);

	if (N->sign == 1) {		// N is a small number, so we make a simple test...
		if (*res = isPrime (N->n[0]))
			sprintf (buf, "%s = %lu is prime! (trial divisions)\n", str, N->n[0]); 
		else
			sprintf (buf, "%s = %lu is not prime. (trial divisions)\n", str, N->n[0]); 
		OutputBoth (buf);
		return (TRUE); 
	}

	if (klen > n) {
	    sprintf(buf, "%s > 2^%lu, so we can only do a PRP test for %s.\n", sgk, n, str);
	    OutputBoth(buf);
		if (is_valid_double (dk))
			retval = fastIsPRP (dk, 2, n, -1, res);
		else if (Nlen < 50) {
			*res = isProbablePrime();
			if (*res)
				sprintf (buf, "%s is a probable prime.\n", str);
			else
				sprintf (buf, "%s is not prime.\n", str);
			OutputBoth (buf);
			retval = TRUE;
		}
		else
			retval = slowIsPRP (str, res);
		if (*res)
			OutputBoth ("Please credit George Woltman's PRP for this result!\n");
		return (retval);
	}

/* Compute the base for the Proth algorithm. */
 
if ((a = genProthBase(gk, n)) < 0) {
	if (a == -1)
		sprintf (buf, "Cannot compute a to test %s...\nThis is surprising, please, let me know that!!\nMy E-mail is jpenne@wanadoo.fr\n", str);
	else
		sprintf (buf, "%s has a small factor : %d !!\n", str, abs(a));
	OutputBoth (buf); 
	*res = FALSE;
	return(TRUE);
}

restart:
	clear_timers ();
	start_timer (0);
	start_timer (1);
	time (&start_time);


/* Assume intermediate results of the length of N. */ 
/* Compute the fftlen for a Mersenne number of this size. */
	
	p = len; 

	fftlmers = gwmap_to_fftlen (1.0, 2, p, -1);

	fftlen = 0;			// Let gwsetup find the fft length

	*res = TRUE;		/* Assume it is a prime */ 

	if (klen < 48 && is_valid_double(dk) && n>=350) {
		ZERO_PADDED_FFT = (dk*a > 8388608.0) ? TRUE : FALSE;
		gwsetup (dk, 2, n, +1, fftlen); 
	}
	else {
		gwsetup_general_mod (N, fftlen);
//		strcpy (GWSTRING_REP, str);
	}


/* Init tmp = (N-1)/2 to compute a^(N-1)/2 mod N */

	tmp = popg ((len >> 5) + 3);
	gtog (N, tmp);
	iaddg (-1, tmp);
	gshiftright (1, tmp);
	Nlen = bitlen (tmp);

/* Init filename */

	tempFileName (filename, 'z');

/* Get the current time */
/* Allocate memory */

	x = gwalloc ();

/* Optionally resume from save file and output a message */
/* indicating we are resuming a test */

	if (fileExists (filename) && readFromFile (filename, &bit, x, NULL)) {
		char	fmt_mask[80];
		double	pct;
		pct = trunc_percent (bit * 100.0 / Nlen);
		sprintf (fmt_mask,
			 "Resuming Proth prime test of %%s at bit %%ld [%%.%df%%%%]\n",
			 PRECISION);
		sprintf (buf, fmt_mask, str, bit, pct);
		OutputStr (buf);
	}

/* Otherwise, output a message indicating we are starting test */

	else {
		sprintf (buf, "Starting Proth prime test of %s\n", str);
		OutputStr (buf);
		bit = 1;
		dbltogw ((double) a, x);
	}

/* Output a message about the FFT length and the Proth base. */

	gwfft_description (fft_desc);
	sprintf (buf, "Using %s, a = %d\n\n", fft_desc, a);
	OutputStr (buf);
	sprintf (buf, "Using %s, a = %d\n", fft_desc, a);
	if (verbose)
		writeResults (buf);
	ReplaceableLine (1);	/* Remember where replaceable line is */

/* Init the title */

	if (strlen (str) < 40)
		title (str);
	else
		title("Running in Big k mode...");

/* Do the Proth test */

//for (int i =0;i < 20; i++) set_fft_value (x, i, 0);
//set_fft_value (x, 16, 200000);
//set_fft_value (x, 16, 0);
//#define CHECK_ITER
#ifdef CHECK_ITER
{giant t1, t2;
t1 = popg ((Nlen >> 4) + 3);
t2 = popg ((Nlen >> 4) + 3);
gwtogiant (x, t1);
#endif
	gwsetmulbyconst (a);
	iters = 0;
	while (bit < Nlen) {

/* Error check the last 50 iterations, before writing an */
/* intermediate file (either user-requested stop or a */
/* 30 minute interval expired), and every 128th iteration. */

		stopping = stopCheck ();
		echk = stopping || ERRCHK || bit+50 >= Nlen;
		if ((bit & 127) == 0) {
			echk = 1;
			time (&current_time);
			saving = (current_time - start_time > write_time);
		} else
			saving = 0;

/* Process this bit */

		gwstartnextfft (!stopping && !saving && bit+26 < Nlen);
#ifdef CHECK_ITER
squareg (t1);
if (bitval (tmp, Nlen-bit-1)) imulg (a, t1);
specialmodg (t1);
gwstartnextfft (0);
echk=1;
#endif
		if (bitval (tmp, Nlen-bit-1)) {
			gwsetnormroutine (0, echk, 1);
		} else {
			gwsetnormroutine (0, echk, 0);
		}
		if (bit+25 < Nlen)
			gwsquare (x);
		else {
			gwsquare_carefully (x);
		}

#ifdef CHECK_ITER
gwtogiant (x, t2);
if (gcompg (t1, t2) != 0)
OutputStr ("Iteration failed.\n");
//if (bit == 100) bit = Nlen;
#endif

/* If the sum of the output values is an error (such as infinity) */
/* then raise an error. */

		if (gw_test_illegal_sumout ()) {
			sprintf (buf, ERRMSG0, bit, Nlen, ERRMSG1A);
			OutputBoth (buf);
			goto error;
		}

/* Check that the sum of the input numbers squared is approximately */
/* equal to the sum of unfft results.  Since this check may not */
/* be perfect, check for identical results after a restart. */

		if (gw_test_mismatched_sums ()) {
			static unsigned long last_bit = 0;
			static double last_suminp = 0.0;
			static double last_sumout = 0.0;
			double suminp, sumout;
			suminp = gwsuminp (x);
			sumout = gwsumout (x);
			if (bit == last_bit &&
			    suminp == last_suminp &&
			    sumout == last_sumout) {
				writeResults (ERROK);
				saving = 1;
			} else {
				char	msg[80];
				sprintf (msg, ERRMSG1B, suminp, sumout);
				sprintf (buf, ERRMSG0, bit, Nlen, msg);
				OutputBoth (buf);
				last_bit = bit;
				last_suminp = suminp;
				last_sumout = sumout;
				goto error;
			}
		}

/* Check for excessive roundoff error  */

		if (echk && MAXERR > 0.40) {
			static unsigned long last_bit = 0;
			static double last_maxerr = 0.0;
			if (bit == last_bit &&
			    MAXERR == last_maxerr) {
				writeResults (ERROK);
				saving = 1;
			} else {
				char	msg[80];
				sprintf (msg, ERRMSG1C, MAXERR);
				sprintf (buf, ERRMSG0, bit, Nlen, msg);
				OutputBoth (buf);
				last_bit = bit;
				last_maxerr = MAXERR;
				goto error;
			}
		}

/* Keep track of maximum and minimum round off error */

		if (ERRCHK) {
			if (MAXERR < reallyminerr && bit > 30)
				reallyminerr = MAXERR;
			if (MAXERR > reallymaxerr)
				reallymaxerr = MAXERR;
		}

/* That iteration succeeded, bump counters */

		bit++;
		iters++;

/* Print a message every so often */

		if (bit % ITER_OUTPUT == 0) {
			char	fmt_mask[80];
			double	pct;
			pct = trunc_percent (bit * 100.0 / Nlen);
			sprintf (fmt_mask, "%%.%df%%%% of %%ld", PRECISION);
			sprintf (buf, fmt_mask, pct, Nlen);
			title (buf);
			ReplaceableLine (2);	/* Replace line */
			sprintf (fmt_mask,
				 "%%s, bit: %%ld / %%ld [%%.%df%%%%]",
				 PRECISION);
			sprintf (buf, fmt_mask, str, bit, Nlen, pct);
			OutputStr (buf);
			if (ERRCHK && bit > 30) {
				OutputStr (".  Round off: ");
				sprintf (buf, "%10.10f", reallyminerr);
				OutputStr (buf);
				sprintf (buf, " to %10.10f", reallymaxerr);
				OutputStr (buf);
			}
			end_timer (0);
			if (CUMULATIVE_TIMING) {
				OutputStr (".  Time thusfar: ");
			} else {
				OutputStr (".  Time per bit: ");
				divide_timer (0, iters);
				iters = 0;
			}
			print_timer (0, TIMER_NL | TIMER_OPT_CLR);
			start_timer (0);
		}

/* Print a results file message every so often */

		if (bit % ITER_OUTPUT_RES == 0 || (NO_GUI && stopping)) {
			sprintf (buf, "Bit %ld / %ld\n", bit, Nlen);
			writeResults (buf);
		}

/* Write results to a file every DISK_WRITE_TIME minutes */
/* On error, retry in 10 minutes (it could be a temporary */
/* disk-full situation) */

		if (saving || stopping) {
			write_time = DISK_WRITE_TIME * 60;
			if (! writeToFile (filename, bit, x, NULL)) {
				sprintf (buf, WRITEFILEERR, filename);
				OutputBoth (buf);
				if (write_time > 600) write_time = 600;
			}
			time (&start_time);

/* If an escape key was hit, write out the results and return */

			if (stopping) {
				pushg (1);
				gwfree (x);
				gwdone ();
				*res = FALSE;		// To avoid credit message !
				return (FALSE);
			}
		}
	}
#ifdef CHECK_ITER
pushg(2);}
#endif

/* See if we've found a Proth prime.  If not, format a 64-bit residue. */
//	trace(0);
	gwtogiant (x, tmp);				// The modulo reduction is done here
//	trace(1);
	iaddg (1, tmp);					// Compute the (unnormalized) residue
//	trace(2);

	if (gcompg (N, tmp) != 0) {
		*res = FALSE;	/* Not a prime */
		if (abs(tmp->sign) < 2)	// make a 32 bit residue correct !!
			tmp->n[1] = 0;
		sprintf (res64, "%08lX%08lX", tmp->n[1], tmp->n[0]);
//		imulg (3, tmp); specialmodg (tmp); ulsubg (3, tmp);
//		sprintf (oldres64, "%08lX%08lX", tmp->n[1], tmp->n[0]);
	}
//	trace(3);
	pushg (1);
//	trace(4);
	gwfree (x);
//	trace(5);

/* Print results.  Do not change the format of this line as Jim Fougeron of */
/* PFGW fame automates his QA scripts by parsing this line. */

	if (*res)
		sprintf (buf, "%s is prime!\n", str);
//	else if (IniGetInt (INI_FILE, "OldRes64", 0))
//		sprintf (buf, "%s is not prime.  RES64: %s.  OLD64: %s\n", str, res64, oldres64);
	else
		sprintf (buf, "%s is not prime.  RES64: %s\n", str, res64);

/* Update the output file */

/*	if ((*res && IniGetInt (INI_FILE, "OutputPrimes", 1)) ||
	    (!*res && IniGetInt (INI_FILE, "OutputComposites", 1)))
		writeResults (buf); */

/* Output the final timings */

	end_timer (1);
	sprintf (buf+strlen(buf)-1, "  Time: ");
	ReplaceableLine (2);	/* Replace line */
	write_timer (buf+strlen(buf), 1, TIMER_CLR | TIMER_NL); 
	OutputBoth (buf);

/* Cleanup and return */

	gwdone ();
	_unlink (filename);
	return (TRUE);

/* An error occured, sleep, then try restarting at last save point. */

error:
	pushg (1);
	gwfree (x);
	gwdone ();

/* Output a message saying we are restarting */

	OutputBoth (ERRMSG2);
	OutputBoth (ERRMSG3);

/* Sleep five minutes before restarting */

	if (! SleepFive ()) return (FALSE);

/* Restart */

	goto restart;
} 

// Some ABC format strings

char ckstring[] = "(2^$a$b)^2-2";		// Carol/Kynea
char cwstring[] = "$a*$b^$a$c";		// Cullen/Woodall
char ffstring[] = "$a*2^$b+1";		// FermFact output

/* Process a number from newpgen output file */
/* NEWPGEN output files use the mask as defined below: */

#define MODE_PLUS    0x01	/* k.b^n+1 */
#define MODE_MINUS   0x02	/* k.b^n-1 */
#define MODE_2PLUS   0x04	/* k.b^(n+1)+1 (*) */
#define MODE_2MINUS  0x08	/* k.b^(n+1)-1 (*) */
#define MODE_4PLUS   0x10	/* k.b^(n+2)+1 (*) */
#define MODE_4MINUS  0x20	/* k.b^(n+2)-1 (*) */
#define MODE_PRIMORIAL 0x40	/* PRIMORIAL - can't handle this */
#define MODE_PLUS5  0x80	/* k.b^n+5 */
#define MODE_AP	    0x200	/* 2^n+2k-1 */
#define MODE_PLUS7  0x800	/* k.b^n+7 */
#define MODE_2PLUS3 0x1000	/* 2k.b^n+3 */
#define MODE_DUAL 0x8000
#define MODE_PLUS_DUAL 0x8001	/* b^n+k */
#define MODE_MINUS_DUAL 0x8002	/* b^n-k */
#define MODE_NOTGENERALISED 0x400

static unsigned __int64 li;

int process_num (
	giant gk,
	char *sgk,
	unsigned long base,
	unsigned long n,
	int	incr,
	int	*res)
{
	char	buf[sgkbufsize+256], str[sgkbufsize+256];
	int	bits, retval;
	double k;

	bits = (int) ((n * log(base)) / log(2) + bitlen(gk)); 
	N = newgiant ((bits >> 4) + 8);

//	Compute the number we are testing.

	ultog (base, N);
	power (N, n);
	mulg (gk, N); 
	iaddg (incr, N);
	Nlen = bitlen(N);

/*	if (!(CPU_FLAGS & CPU_SSE2)) {
		sprintf (buf, "Sorry, this program actually works only on SSE2 machines.\n"); 
		OutputBoth (buf);
		Sleep(10000);				// Wait 10 seconds before exiting...
		free (N); 
		term_giants (); 
		exit(0);
	} */

	if (base == 2 && !IniGetInt (INI_FILE, "ForcePRP", 0) && ((incr == -1) || (incr == +1))) {
		giant gk_copy = newgiant(2*(abs(gk->sign) + 1));
		gtog (gk, gk_copy);
		if (incr == -1)
			retval = isLLRP_p (gk_copy, sgk, n, res);
		else
			retval = isProthP_p (gk_copy, sgk, n, res);
		free (gk_copy);
		free (N);
		return (retval);
	} else {
		if (mask & MODE_DUAL) {
				sprintf (str, "%lu^%lu%c%d", base, n, incr < 0 ? '-' : '+', abs(incr));
		}
		else if (incr != -2) {	// Not MODE_AP
			if (isone(gk))
				sprintf (str, "%lu^%lu%c%d", base, n, incr < 0 ? '-' : '+', abs(incr));
		    else
				sprintf (str, "%s*%lu^%lu%c%d", sgk, base, n, incr < 0 ? '-' : '+', abs(incr));
		}
		else {					// MODE_AP
		    incr = -1;
		    if (isone(gk))
				sprintf (str, "%lu^%lu+1", base, n);
		    else
				sprintf (str, "%lu^%lu+2*%s-1", base, n, sgk);

//			Recomputing the number we are testing.			
			
			ultog (base, N); 
		    power (N, n); 
		    addg(gk, N);
		    addg(gk, N);
		    iaddg (incr, N); 
			Nlen = bitlen(N);
		}
		sprintf (buf, "LLR tests only k*2^n1 numbers, so, we will do a PRP test of %s\n", str); 
		OutputBoth (buf); 
		k = atof (sgk);
		if (bitlen (gk) < 48 && is_valid_double (k))
			retval = fastIsPRP (k, base, n, incr, res);
		else if (Nlen < 50) {
			*res = isProbablePrime();
			if (*res)
				sprintf (buf, "%s is a probable prime.\n", str);
			else
				sprintf (buf, "%s is not prime.\n", str);
			OutputBoth (buf);
			retval = TRUE;
		}
		else
			retval = slowIsPRP (str, res);
		if (*res)
			OutputBoth ("Please credit George Woltman's PRP for this result!\n");
		free (N);
		return (retval);
	}
}



void primeContinue ()
{

	int	workdone, work;
	unsigned int iscw = 0, isff = 0, isck = 0;
	char *pinput;

/* If we are done, return */

	workdone = IniGetInt (INI_FILE, "WorkDone", 1);

	if (workdone == 1) return;

/* Set appropriate priority */

	SetPriority ();

/* Case off the work type */

	work = IniGetInt (INI_FILE, "Work", 0);

/* Handle a newpgen output file */

	if (work == 0) {
	    char	inputfile[80], outputfile[80], sgk[sgkbufsize], buff[sgkbufsize+256];
		char	outbuf[sgkbufsize+256];
	    FILE *fd;
	    unsigned long i, chainlen, n, base, nfudge, nn;
	    giant gk, gkk, g1;
	    int	firstline, line, outfd, res, uptoline, incr;
	    char c;

	    IniGetString (INI_FILE, "PgenInputFile", inputfile, 80, NULL);
	    IniGetString (INI_FILE, "PgenOutputFile", outputfile, 80, NULL);
	    firstline = IniGetInt (INI_FILE, "PgenLine", 1);
	    verbose = IniGetInt (INI_FILE, "Verbose", 0);

	    fd = fopen (inputfile, "r");
	    if (fd == NULL) {
		IniWriteInt (INI_FILE, "WorkDone", 1);
		return;
	    }

		uptoline = IniGetInt(INI_FILE, "VerifyUpToLine", 0);

		if (fgets (buff, sgkbufsize+128, fd) == NULL) {
		IniWriteInt (INI_FILE, "WorkDone", 1);
		return;
	    }

		// Special code to process some ABC file formats...

		testcarol = testkynea = FALSE;

		if (!strncmp (buff, "ABC", 3)) {
			if (! fileExists (outputfile)) {
				outfd = _open (outputfile, _O_TEXT | _O_RDWR | _O_CREAT, 0666);
				if (outfd) {
				_write (outfd, buff, strlen (buff));
				_close (outfd);
				}
			}

			g1 = newgiant (2);
			setone (g1);					// One in a giant...
			for (pinput=buff+3; *pinput && isspace(*pinput); pinput++);

			if (!strncmp (pinput, cwstring, strlen (cwstring))) {
				iscw = TRUE;
				gk = newgiant (2);
				gkk = newgiant (2);
			}
			else if (!strncmp (pinput, ffstring, strlen (ffstring))) {
				isff = TRUE;
				gk = newgiant (sgkbufsize);
				gkk = newgiant (sgkbufsize);
			}
			else if (!strncmp (pinput, ckstring, strlen (ckstring))) {
				isck = TRUE;
				gk = newgiant (sgkbufsize);
				gkk = newgiant (sgkbufsize);
			}
			else {
				sprintf (outbuf, "%s : Unknown ABC format.\n", buff);
				OutputStr (outbuf);
				IniWriteInt (INI_FILE, "WorkDone", 1);
				return;
			}


/* Process each line in the output file */

			for (line = 1; ; line++) {

/* Read the line, break at EOF */

				if (fgets (buff, sgkbufsize+256, fd) == NULL) {
					IniWriteInt (INI_FILE, "WorkDone", 1);
					break;
				}

/* Skip this line if requested (we processed it on an earlier run) */

				if (line < firstline) continue;

				if (iscw) {						// Cullen/Woodall
					if (sscanf (buff, "%lu %lu %d", &n, &base, &incr) != 3)
						continue;				// Skip invalid line
					ultog (n, gk);
					sprintf (sgk, "%lu", n);
					if (! process_num (gk, sgk, base, n, incr, &res)) goto done;
					if (res) {
						outfd = _open (outputfile, _O_TEXT | _O_RDWR | _O_APPEND | _O_CREAT, 0666);
						if (outfd) {
							sprintf (outbuf, "%lu %lu %d\n", n, base, incr); 
							_write (outfd, outbuf, strlen (outbuf));
							_close (outfd);
						}
					}
				}
				else if (isff)	{				// FermFact output
												// allow k to be a big integer
					if (sscanf (buff, "%s %lu", sgk, &n) != 2)
						continue;				// Skip invalid line
					if (!isDigitString(sgk))
						continue;				// Skip invalid line
					ctog(sgk, gk);
					if (! process_num (gk, sgk, 2, n, +1, &res)) goto done;
					if (res) {
						outfd = _open (outputfile, _O_TEXT | _O_RDWR | _O_APPEND | _O_CREAT, 0666);
						if (outfd) {
							sprintf (outbuf, "%s %lu\n", sgk, n); 
							_write (outfd, outbuf, strlen (outbuf));
							_close (outfd);
						}
					}
				}
				else {							// Carol/Kynea
					if (sscanf (buff, "%lu %d", &n, &incr) != 2)
						continue;						// Skip invalid line
					testkynea = (incr == 1) ? TRUE : FALSE;
					testcarol = (incr == -1) ? TRUE : FALSE;
					free (gk);
					free (gkk);
					gk = newgiant ((n>>4)+1);
					gkk = newgiant ((n>>4)+1);			// Useless...
					setone (gk);						// Compute k multiplier
					gshiftleft (n-1, gk);
					if (testkynea) {
						addg(g1, gk);
						sprintf (sgk, "(2^%lu+1)", n-1);
//						sprintf (nstring, "%s*2^%lu-1 = (2^%lu+1)^2 - 2",sgk, n+1, n);
					}
					else if (testcarol) {
						subg(g1, gk);
						sprintf (sgk, "(2^%lu-1)", n-1);
//						sprintf (nstring, "%s*2^%lu-1 = (2^%lu-1)^2 - 2",sgk, n+1, n);
					}
					else
						continue;
					if (! process_num (gk, sgk, 2, n+1, -1, &res)) goto done;
					if (res) {
						outfd = _open (outputfile, _O_TEXT | _O_RDWR | _O_APPEND | _O_CREAT, 0666);
						if (outfd) {
							sprintf (outbuf, "%lu %d\n", n, incr); 
							_write (outfd, outbuf, strlen (outbuf));
							_close (outfd);
						}
					}
				}
				IniWriteInt (INI_FILE, "PgenLine", line + 1);
			}
			goto done;
		}

	    mask = 0;
	    sscanf (buff, $LLF":%c:%lu:%lu:%lu\n", &li, &c, &chainlen, &base, &mask);

	    if (mask & 0x40) {
		OutputStr ("Primoral NewPgen files are not supported.\n");
		return;
	    }
	    if (chainlen == 0) chainlen = 1;

	    if (! fileExists (outputfile)) {
		outfd = _open (outputfile, _O_TEXT | _O_RDWR | _O_CREAT, 0666);
		if (outfd) {
			if (mask == 0)
				sprintf (outbuf, $LLF":%c:%lu:%lu\n", li, c, chainlen, base);
			else
				sprintf (outbuf, $LLF":%c:%lu:%lu:%lu\n", li, c, chainlen, base, mask);
			_write (outfd, outbuf, strlen (outbuf));
			_close (outfd);
		}
	    }

		gk = newgiant (sgkbufsize);
		gkk = newgiant (sgkbufsize);
		g1 = newgiant (2);
		setone (g1);					// One in a giant...

/* THIS SECTION IS FOR BACKWARDS COMPATIBILITY WITH PREVIOUS PRP.EXE */
/* That version used the one character code to determine what to do. */
/* The new version uses the mask field. */

	    if (mask == 0 || IniGetInt (INI_FILE, "UseCharCode", 0)) {

/* The variable c is a one character code as follows: */
/* P : k.b^n+1 (Plus)
   M : k.b^n-1 (Minus)
   T: k.b^n+-1 (Twin)
   S: k.b^n-1; k.b^(n+1)-1 (SG (CC 1st kind len 2))
   C: k.b^n+1; k.b^(n+1)+1 (CC 2nd kind len 2)
   B: k.b^n+-1; k.b^(n+1)+-1 (BiTwin)
   J: k.b^n+-1; k.b^(n+1)-1 (Twin/SG)
   K: k.b^n+-1; k.b^(n+1)+1 (Twin/CC)
   Y : k.b^n+1 + others (Lucky Plus)
   Z : k.b^n-1 + others (Lucky Minus)
   1: CC 1st kind chain
   2: CC 2nd kind chain
   3: BiTwin chain */
/* Undo the increment of n that newpgen did on types 1, 2, 3 */
/* Map P, M, Y, Z, T, S, C, B to their more generic counterparts */

		nfudge = 0;
		if (c == '1') nfudge = 1;
		if (c == '2') nfudge = 1;
		if (c == '3') nfudge = 1;
		if (c == 'P') c = '2', chainlen = 1;
		if (c == 'M') c = '1', chainlen = 1;
//		if (c == 'Y') c = '2', chainlen = 1;
//		if (c == 'Z') c = '1', chainlen = 1;
		if (c == 'T') c = '3', chainlen = 1;
		if (c == 'S') c = '1', chainlen = 2;
		if (c == 'C') c = '2', chainlen = 2;
		if (c == 'B') c = '3', chainlen = 2;

/* Process each line in the newpgen output file */

		for (line = 1; ; line++) {

/* Read the line, break at EOF */

			if (fgets (buff, sgkbufsize+256, fd) == NULL) {
			IniWriteInt (INI_FILE, "WorkDone", 1);
			break;
		    }

/* Skip this line if requested (we processed it on an earlier run) */

			if (line < firstline) continue;

					// allow k to be a big integer
			if (sscanf (buff, "%s %lu", sgk, &n) != 2)
				continue;						// Skip invalid line

			if (!isDigitString(sgk)) continue;	// Skip invalid line

			ctog(sgk, gk);

			skiptest = (line <= uptoline);

/* Test numbers according to the c variable */

			nn = n;
			if (c == 'Y') {
				nn--;
			}
			if (c == 'Z') {
				nn--;
			}

			for (i = 0; i < chainlen; i++) {
				if (c == '1' || c == '3') {
					if (! process_num (gk, sgk, base, n - nfudge + i, -1, &res)) goto done;
					if (!res) break;
				}
				if (c == '2' || c == '3') {
					if (! process_num (gk, sgk, base, n - nfudge + i, +1, &res)) goto done;
					if (!res) break;
				}
				if (c == 'J') {	// Twin/SG
					int	res2;
					if (! process_num (gk, sgk, base, n, -1, &res)) goto done;
					if (!res) break;
					if (! process_num (gk, sgk, base, n+1, -1, &res)) goto done;
					if (! process_num (gk, sgk, base, n, +1, &res2)) goto done;
					res |= res2;
					break;
				}
				if (c == 'K') {	// Twin/CC
					int	res2;
					if (! process_num (gk, sgk, base, n, +1, &res)) goto done;
					if (!res) break;
					if (! process_num (gk, sgk, base, n, -1, &res)) goto done;
					if (! process_num (gk, sgk, base, n+1, +1, &res2)) goto done;
					res |= res2;
					break;
				}
				if (c == 'Y') {	// Lucky Plus
					int	res2;
					if (! process_num (gk, sgk, base, nn, +1, &res)) goto done;
					if (!res) break;
					if (! process_num (gk, sgk, base, nn+1, -1, &res)) goto done;
					if (! process_num (gk, sgk, base, nn+1, +1, &res2)) goto done;
					res |= res2;
					if (! process_num (gk, sgk, base, nn+2, +1, &res2)) goto done;
					res |= res2;
					break;
				}
				if (c == 'Z') {	// Lucky Minus
					int	res2;
					if (! process_num (gk, sgk, base, nn, -1, &res)) goto done;
					if (!res) break;
					if (! process_num (gk, sgk, base, nn+1, -1, &res)) goto done;
					if (! process_num (gk, sgk, base, nn+1, +1, &res2)) goto done;
					res |= res2;
					if (! process_num (gk, sgk, base, nn+2, -1, &res2)) goto done;
					res |= res2;
					break;
				}
				if (c == 'A') {
					if (! process_num (gk, sgk, base, n, -2, &res)) goto done;
					if (!res) break;
				}
			}

/* If all numbers tested were probable primes, copy the line to the output file */
			if (res) {
				outfd = _open (outputfile, _O_TEXT | _O_RDWR | _O_APPEND | _O_CREAT, 0666);
				if (outfd) {
					sprintf (outbuf, "%s %lu\n", sgk, n); 
					_write (outfd, outbuf, strlen (outbuf));
					_close (outfd);
				}
			}
			IniWriteInt (INI_FILE, "PgenLine", line + 1);
		}


/* THIS IS THE NEW SECTION.  It uses both the mask field and the */
/* character code to determine what to do */

	    } else {
//		unsigned long nn;

/* NEWPGEN output files use the mask as defined below: */
/* #define MODE_PLUS    0x01	/* k.b^n+1 */
/* #define MODE_MINUS   0x02	/* k.b^n-1 */
/* #define MODE_2PLUS   0x04	/* k.b^(n+1)+1 (*) */
/* #define MODE_2MINUS  0x08	/* k.b^(n+1)-1 (*) */
/* #define MODE_4PLUS   0x10	/* k.b^(n+2)+1 (*) */
/* #define MODE_4MINUS  0x20	/* k.b^(n+2)-1 (*) */
/* #define MODE_PRIMORIAL 0x40	/* PRIMORIAL - can't handle this */
/* #define MODE_PLUS5  0x80	/* k.b^n+5 */
/* #define MODE_AP	    0x200	/* 2^n+2k-1 */
/* #define MODE_PLUS7  0x800	/* k.b^n+7 */
/* #define MODE_2PLUS3 0x1000	/* 2k.b^n+3 */
/* #define MODE_DUAL 0x8000
/* #define MODE_PLUS_DUAL 0x8001	/* b^n+k */
/* #define MODE_MINUS_DUAL 0x8002	/* b^n-k */
/* #define MODE_NOTGENERALISED 0x400
/* Those entries that have a (*) next to them are modified if the */
/* MODE_NOTGENERALISED flag is set.  If it is set, they are changed */
/* as follows: */
/* MODE_2PLUS      2k.b^n+1 */
/* MODE_2MINUS     2k.b^n-1 */
/* MODE_4PLUS      4k.b^n+1 */
/* MODE_4MINUS     4k.b^n-1 */
/* Similarly, longer chains are affected in the same way (so if the base */
/* is 3 and we are after a CC of the 1st kind of length 4, rather that */
/* looking at k.3^n-1 & k.3^(n+1)-1 & k.3^(n+2)-1 & k.3^(n+3)-1 we look */
/* at k.3^n-1 & 2k.3^n-1 & 4k.3^n-1 & 8k.3^n-1). */

/* Process each line in the newpgen output file */

		for (line = 1; ; line++) {

/* Read the line, break at EOF */

			if (fgets (buff, sgkbufsize+256, fd) == NULL) {
			IniWriteInt (INI_FILE, "WorkDone", 1);
			break;
		    }

/* Skip this line if requested (we processed it on an earlier run) */

			if (line < firstline) continue;
					// allow k to be a big integer
			if (sscanf (buff, "%s %lu", sgk, &n) != 2)
				continue;						// Skip invalid line

			if (!isDigitString(sgk)) continue;	// Skip invalid line

			ctog(sgk, gk);

			skiptest = (line <= uptoline);

/* Undo the increment of n that newpgen did on types 1, 2, 3 */

			nn = n;
			if (c == '1' || c == '2' || c == '3') nn--;

			if ((mask & MODE_PLUS) && (mask & MODE_2MINUS) && (mask & MODE_2PLUS) && (mask & MODE_4PLUS)) {
				nn--;
			}
			if ((mask & MODE_MINUS) && (mask & MODE_2MINUS) && (mask & MODE_2PLUS) && (mask & MODE_4MINUS)) {
				nn--;
			}

/* Test numbers according to the mask variable */
/* The J and K types (Twin/CC and Twin/SG) are special in that they */
/* are output if either a Twin OR a CC/SG is found */

			gtog(gk, gkk);

			for (i = 0; i < chainlen; i++) {
				gtoc(gkk, sgk, sgkbufsize);
				if ((mask & MODE_MINUS) && (mask & MODE_PLUS) && (mask & MODE_2MINUS)) {	// Twin/SG
					int	res2;
					if (! process_num (gkk, sgk, base, nn, -1, &res)) goto done;
					if (!res) break;
					if (! process_num (gkk, sgk, base, nn+1, -1, &res)) goto done;
					if (! process_num (gkk, sgk, base, nn, +1, &res2)) goto done;
					res |= res2;
					break;
				}
				if ((mask & MODE_MINUS) && (mask & MODE_PLUS) && (mask & MODE_2PLUS)) {	// Twin/CC
					int	res2;
					if (! process_num (gkk, sgk, base, nn, +1, &res)) goto done;
					if (!res) break;
					if (! process_num (gkk, sgk, base, nn, -1, &res)) goto done;
					if (! process_num (gkk, sgk, base, nn+1, +1, &res2)) goto done;
					res |= res2;
					break;
				}
				if ((mask & MODE_PLUS) && (mask & MODE_2MINUS) && (mask & MODE_2PLUS) && (mask & MODE_4PLUS)) {	// Lucky Plus
					int	res2;
					if (! process_num (gk, sgk, base, nn, +1, &res)) goto done;
					if (!res) break;
					if (! process_num (gk, sgk, base, nn+1, -1, &res)) goto done;
					if (! process_num (gk, sgk, base, nn+1, +1, &res2)) goto done;
					res |= res2;
					if (! process_num (gk, sgk, base, nn+2, +1, &res2)) goto done;
					res |= res2;
					break;
				}
				if ((mask & MODE_MINUS) && (mask & MODE_2MINUS) && (mask & MODE_2PLUS) && (mask & MODE_4MINUS)) {	// Lucky Minus
					int	res2;
					if (! process_num (gk, sgk, base, nn, -1, &res)) goto done;
					if (!res) break;
					if (! process_num (gk, sgk, base, nn+1, -1, &res)) goto done;
					if (! process_num (gk, sgk, base, nn+1, +1, &res2)) goto done;
					res |= res2;
					if (! process_num (gk, sgk, base, nn+2, -1, &res2)) goto done;
					res |= res2;
					break;
				}
				if (mask & MODE_MINUS) {
					if (mask & MODE_DUAL) {
						if (! process_num (g1, "1", base, nn, -(long)gkk->n[0], &res)) goto done;
					}
					else
						if (! process_num (gkk, sgk, base, nn, -1, &res)) goto done;
					if (!res) break;
				}
				if (mask & MODE_PLUS) {
					if (mask & MODE_DUAL) {
						if (! process_num (g1, "1", base, nn, gkk->n[0], &res)) goto done;
					}
					else
						if (! process_num (gkk, sgk, base, nn, +1, &res)) goto done;
					if (!res) break;
				}
				if (mask & MODE_PLUS5) {
					if (! process_num (gkk, sgk, base, nn, +5, &res)) goto done;
					if (!res) break;
				}
				if (mask & MODE_PLUS7) {
					if (! process_num (gkk, sgk, base, nn, +7, &res)) goto done;
					if (!res) break;
				}
				if (mask & MODE_2PLUS3) {
					gshiftleft(1, gkk);	// Compute gkk + gkk
					gtoc(gkk, sgk, sgkbufsize);
					if (! process_num (gkk, sgk, base, nn, +3, &res)) goto done;
					gshiftright(1, gkk);	// Restore gkk
					gtoc(gkk, sgk, sgkbufsize);
					if (!res) break;
				}
				if (mask & MODE_AP) {
					if (! process_num (gkk, sgk, base, nn, -2, &res)) goto done;
					if (!res) break;
				}

/* Bump k or n for the next iteration or for the MODE_2PLUS and */
/* MODE_2MINUS flags */

				if (mask & MODE_NOTGENERALISED) gshiftleft(1, gkk); 
				else nn += 1;
				gtoc(gkk, sgk, sgkbufsize);

/* If chainlength is more than 1, then we let the for loop do the work */
/* rather than the MODE_2PLUS, etc. flags */

				if (chainlen > 1) continue;

				if (mask & MODE_2MINUS) {
					if (! process_num (gkk, sgk, base, nn, -1, &res)) goto done;
					if (!res) break;
				}
				if (mask & MODE_2PLUS) {
					if (! process_num (gkk, sgk, base, nn, +1, &res)) goto done;
					if (!res) break;
				}

/* Bump k or n for the MODE_4PLUS and MODE_4MINUS flags */

				if (mask & MODE_NOTGENERALISED) gshiftleft(1, gkk); 
				else nn += 1;
				gtoc(gkk, sgk, sgkbufsize);

				if (mask & MODE_4MINUS) {
					if (! process_num (gkk, sgk, base, nn, -1, &res)) goto done;
					if (!res) break;
				}
				if (mask & MODE_4PLUS) {
					if (! process_num (gkk, sgk, base, nn, +1, &res)) goto done;
					if (!res) break;
				}
			}

/* If all numbers tested were probable primes, copy the line to the output file */

			if (res) {
				gtoc(gk, sgk, sgkbufsize);
				outfd = _open (outputfile, _O_TEXT | _O_RDWR | _O_APPEND | _O_CREAT, 0666);
				if (outfd) {
					sprintf (outbuf, "%s %lu\n", sgk, n); 
					_write (outfd, outbuf, strlen (outbuf));
					_close (outfd);
				}
			}
			IniWriteInt (INI_FILE, "PgenLine", line + 1);
		}

		
	    }

done:	    fclose (fd);
			free(gk);
			free(gkk);
			free(g1);
			term_giants();
	}

/* Handle an expr */

	else {
		OutputStr ("Expression testing not yet implemented.\n");
		IniWriteInt (INI_FILE, "WorkDone", 1);
	}
}
